/**
 * Copyright (C) 2001, 2002, 2003 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * @author bche@redhat.com
 */

package com.arsdigita.populate.cms;

import com.arsdigita.cms.ContentBundle;
import com.arsdigita.cms.ContentItem;
import com.arsdigita.cms.ContentPage;
import com.arsdigita.cms.ContentSection;
import com.arsdigita.cms.ContentTypeLifecycleDefinition;
import com.arsdigita.cms.lifecycle.LifecycleDefinition;
import com.arsdigita.util.UncheckedWrapperException;
import com.arsdigita.workflow.simple.TaskCollection;
import com.arsdigita.workflow.simple.Workflow;
import com.arsdigita.workflow.simple.WorkflowTemplate;
import org.apache.log4j.Logger;

import java.util.Date;


public abstract class CreateContentPage 
    extends AbstractCreateContentItem {

    private static final Logger s_log =
        Logger.getLogger(CreateContentPage.class.getName());

    private ContentPage m_item = null;
    private ContentBundle m_bundle = null;
    private Class m_itemClass = null;
    private String m_name = null;

    private String m_sIdentifier;
    private ContentSection m_section = null;
    private ContentItem m_parent = null;
    
    public CreateContentPage(Class itemClass, String name) {
        m_itemClass = itemClass;
        m_name = name;
    }

    /**
     * @see CreateContentItem#setup(String, ContentItem, ContentSection)
     */
    public void setup(String sIdentifier, ContentItem parent,
                      ContentSection section) {
        m_sIdentifier = sIdentifier;
        m_parent = parent;
        m_section = section;
    }

    /**
     * @see CreateContentItem#makeContent()
     */
    public abstract void makeContent();

    /**
     * Publishes now.
     *
     * @see #publish(Date)
     **/
    public void publish() {
        publish(null);
    }

    /**
     * @see CreateContentItem#publish()
     */
    public void publish(Date pubDate) {
        //////////////////////////////////
        //apply workflow
        //////////////////////////////////
        //get the workflow template
        TaskCollection workflowTemplates = getSection().getWorkflowTemplates();
        //by default, an article has one workflow, which is "production workflow"
        workflowTemplates.next();
        WorkflowTemplate workflowTemplate =
            (WorkflowTemplate) workflowTemplates.getTask();
        //s_log.info("workflow template is " + workflowTemplate.getLabel());
        workflowTemplates.close();

        //create a workflow based on the workflow template
        Workflow w = workflowTemplate.instantiateNewWorkflow();
        w.setObject(m_item);
        finishTasks(w);

        //////////////////////////////////
        //make it live
        //////////////////////////////////

        //get the lifecycle def
        LifecycleDefinition def =
            ContentTypeLifecycleDefinition.getLifecycleDefinition(
                m_section,
                m_item.getContentType());

        //publish now
        m_item.publish(def, pubDate);
    }

    /**
     * @see CreateContentItem#getContentType()
     */
    public String getContentType() {
        return m_item.getContentType().getClass().getName();
    }

    /**
     * @see CreateContentItem#getContentTypeName()
     */
    public String getContentTypeName() {
        return m_name;
    }

    /**
     * @see CreateContentItem#getContentItem()
     */
    public ContentItem getContentItem() {
        return m_item;
    }

    public ContentPage getContentPage() {
        return m_item;
    }

    protected ContentBundle getContentBundle() {
        return m_bundle;
    }
    
    protected ContentItem getParent() {
        return m_parent;
    }
    
    protected ContentSection getSection() {
        return m_section;
    }

    protected ContentPage instantiateItem() {
        ContentPage item = null;
        try {
            item = (ContentPage)m_itemClass.newInstance();
        } catch (IllegalAccessException ex) {
            throw new UncheckedWrapperException(
                "cannot create item of type " + m_itemClass.getName(), ex);
        } catch (InstantiationException ex) {
            throw new UncheckedWrapperException(
                "cannot create item of type " + m_itemClass.getName(), ex);
        }
        return item;
    }

    /**
     * Creates a basic multi-lingual item in english
     */
    protected ContentPage createItem() {
        return createItem("en");
    }

    /**
     * Creates a basic multi-lingual item.
     * @param language the items language
     */
    protected ContentPage createItem(String language) {
        m_item = instantiateItem();
        m_item.setTitle(m_sIdentifier);
        m_item.setName(m_sIdentifier);
        m_item.setLaunchDate(new Date());
        m_item.setLanguage(language);
        
        m_bundle = new ContentBundle(m_item);
        m_bundle.setParent(m_parent);
        m_bundle.save();
        
        return m_item;
    }    
}
