/*
 * Copyright (C) 2001, 2002 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 */

package com.arsdigita.populate.cms;

import com.arsdigita.cms.ContentSection;
import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.kernel.KernelExcursion;
import com.arsdigita.kernel.SiteNode;
import com.arsdigita.kernel.User;
import com.arsdigita.kernel.UserAuthentication;
import com.arsdigita.kernel.UserCollection;
import com.arsdigita.kernel.permissions.PermissionService;
import com.arsdigita.kernel.permissions.PrivilegeDescriptor;
import com.arsdigita.kernel.permissions.UniversalPermissionDescriptor;
import com.arsdigita.persistence.Session;
import com.arsdigita.persistence.SessionManager;
import com.arsdigita.persistence.TransactionContext;
import com.arsdigita.runtime.AbstractScript;
import com.arsdigita.runtime.InteractiveParameterLoader;
import com.arsdigita.runtime.Script;
import com.arsdigita.runtime.ScriptContext;
import com.arsdigita.runtime.Startup;
import com.arsdigita.util.StringUtils;
import com.arsdigita.util.parameter.ClassParameter;
import com.arsdigita.util.parameter.ErrorList;
import com.arsdigita.util.parameter.IntegerParameter;
import com.arsdigita.util.parameter.Parameter;
import com.arsdigita.util.parameter.ParameterError;
import com.arsdigita.util.parameter.ParameterLoader;
import com.arsdigita.util.parameter.ParameterReader;
import com.arsdigita.util.parameter.ParameterWriter;
import com.arsdigita.util.parameter.StringParameter;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import org.apache.log4j.Logger;
import org.apache.oro.text.perl.Perl5Util;

/**
 * @author bche
 */
public class CMSLoader extends AbstractScript {
    private static final Logger s_log = Logger.getLogger(CMSLoader.class);

    private static final String PARAM_PREFIX = "waf.populate.cms.";

    private static final String POP_USER_EMAIL="population-user@redhat.rhat";
    private static final String POP_USER_FIRST_NAME="population";
    private static final String POP_USER_LAST_NAME="user";

    private final Parameter m_baseStringSeed;
    private final Parameter m_contentSection;
    private final Parameter m_numTopLevelFolders;
    private final Parameter m_folderDepth;
    private final Parameter m_numContentItems;
    private final Parameter m_createContentTypes;
    private final Parameter m_numUsers;

    public CMSLoader() {
        m_baseStringSeed = new StringParameter(PARAM_PREFIX + "base_string_seed");
        register(m_baseStringSeed);
        m_contentSection = new StringParameter(PARAM_PREFIX + "content_section");
        register(m_contentSection);
        m_numTopLevelFolders= new IntegerParameter(PARAM_PREFIX + "num_top_level_folders");
        register(m_numTopLevelFolders);
        m_folderDepth = new IntegerParameter(PARAM_PREFIX + "folder_depth");
        register(m_folderDepth);
        m_numContentItems = new IntegerParameter(PARAM_PREFIX + "num_content_items");
        register(m_numContentItems);
        m_createContentTypes = new ClassArrayParameter(PARAM_PREFIX + "create_content_types");
        register(m_createContentTypes);
        m_numUsers = new IntegerParameter(PARAM_PREFIX + "num_users");
        register(m_numUsers);
    }

    /**
     * creates or retrieves the population user
     * @return the population user
     */
    private User getPopulationUser() {
        User u = null;

        String sContentSection = (String)get(m_contentSection);
        ContentSection section = getContentSection(sContentSection);

        Session ses = SessionManager.getSession();
        TransactionContext txn = ses.getTransactionContext();

        txn.beginTxn();

        UserCollection  users = User.retrieveAll();
        users.addEqualsFilter("primaryEmail", POP_USER_EMAIL);

        if (users.next()) {
            if (s_log.isDebugEnabled()) {
                s_log.debug("retrieving population user from database");
            }

            u = users.getUser();
        } else {
            if (s_log.isDebugEnabled()) {
                s_log.debug("creating new population user");
            }

            u = new User(POP_USER_FIRST_NAME, POP_USER_LAST_NAME, POP_USER_EMAIL);
            u.save();

            //set password
            UserAuthentication auth = null;
            auth = UserAuthentication.createForUser(u);
            auth.setPassword("redhat");
            auth.setPasswordQuestion("redhat");
            auth.setPasswordAnswer("redhat");
            auth.save();
        }

        users.close();
        txn.commitTxn();
        return u;
    }

    /* (non-Javadoc)
     * @see com.arsdigita.runtime.Script#run(com.arsdigita.runtime.ScriptContext)
     */
    public void run(final ScriptContext context) {
        //run in a kernel context to set the party and effective party
        new KernelExcursion() {
            protected void excurse() {
                User u = null;
                try {
                    u = getPopulationUser();
                    //grant user admin permission for population
                    s_log.info("GRANTING POPULATION USER SITE-WIDE ADMIN FOR POPULATION...");
                    PermissionService.grantPermission(
                        new UniversalPermissionDescriptor(PrivilegeDescriptor.ADMIN, u));
                    setEffectiveParty(u);
                    setParty(u);

                    doRun(context);
                } finally {
                    //revoke user permission after population to avoid security hole
                    if (u != null) {
                        s_log.info("REVOKING SITE-WIDE ADMIN FROM  POPULATION USER AFTER  POPULATION...");
                        PermissionService.revokePartyPermissions(u.getOID());
                    }
                }
            }
        }.run();
    }

    private void doRun(ScriptContext context) {
        //disable p2fs and lifecycle during population
        s_log.info("disabling PublishToFilesystem and Lifecycle during CMS population...");
        com.arsdigita.cms.publishToFile.QueueManager.stopWatchingQueue();
        com.arsdigita.cms.lifecycle.Scheduler.stopTimer();

        String sBaseStringSeed = (String)get(m_baseStringSeed);
        int iTopFolders = ((Integer)get(m_numTopLevelFolders)).intValue();
        int iFolderDepth = ((Integer)get(m_folderDepth)).intValue();
        int iContentItems = ((Integer)get(m_numContentItems)).intValue();
        String sContentSection = (String)get(m_contentSection);
        Class[] createContentTypes = (Class[])get(m_createContentTypes);
        Integer iUsers = (Integer)get(m_numUsers);

        s_log.info("Begin populating content items at " + (new Date()).toString() + "...");
        final long start = System.currentTimeMillis();

        if (s_log.isDebugEnabled()) {
            s_log.debug("Using BaseStringSeed " + sBaseStringSeed);
        }

        //get the content section
        s_log.info("getting content section " + sContentSection);
        ContentSection section = getContentSection(sContentSection);

        if (section == null) {
            throw new NullPointerException("The content section " + sContentSection + " does not exist");
        }

        //create the folders
        PopulateFolders popFolders = new PopulateFolders();
        popFolders.setBaseStringSeed(sBaseStringSeed);
        s_log.info("creating " + iTopFolders + " top-level folders with a depth of " +
                          iFolderDepth);
        popFolders.populate(iTopFolders, iFolderDepth, section);
        s_log.info("finished creating folders");

        //get the folders.  we add the root folder to stick things in there as well
        FolderList folders = popFolders.getFolders();
        folders.add(0, section.getRootFolder());
        if (s_log.isDebugEnabled()) {
            s_log.debug("using " + folders.size() + " folders for item population");
        }

        //create the content items
        s_log.info("creating " + iContentItems + " content items each for " + createContentTypes.length  +
                        " content types");
        PopulateContent pop = new PopulateContent(iContentItems, section);
        pop.setBaseStringSeed(sBaseStringSeed);
        int iParentIndex = 0;
        for (int i=0; i < createContentTypes.length; i++) {
            Class createItemClass = createContentTypes[i];
            CreateContentItem createItem = null;
            try {
                createItem = (CreateContentItem) createItemClass.newInstance();
            } catch (Exception e) {
                throw new RuntimeException("error getting createItem " + e.getMessage());
            }

            pop.populate(createItem, folders, iParentIndex);
            iParentIndex = pop.getLastFolderIndex() + 1;
        }
        s_log.info("Finished populating content items at " + (new Date()).toString() + ".");

        if (iContentItems > 0) {
            final long elapsed = System.currentTimeMillis() - start;
            final long per =
                elapsed / (iContentItems * createContentTypes.length);

            s_log.info
                ("Elapsed millis " + elapsed + " (" + per + " per item)");
        }

        //create users
        s_log.info("creating " + iUsers.intValue() + " users");
        PopulateUsers popUsers = new PopulateUsers();
        popUsers.setBaseStringSeed(sBaseStringSeed);
        s_log.info("Begin populating users at " + (new Date()).toString() + ".");
        popUsers.populate(iUsers.intValue(), section);
        s_log.info("Finished populating users at " + (new Date()).toString() + ".");

        //restart p2fs and lifecycle again to process populated items
        s_log.info("Restarting lifecycle and p2fs processes...");
        com.arsdigita.cms.lifecycle.Scheduler.startTimer();
        //we can't get the original QueueManager pollDelay settings, so use the default of 5 seconds
        com.arsdigita.cms.publishToFile.QueueManager.startWatchingQueue(1,5);
    }

    /*
     * gets the content section with the name, sSection.
     */
    private ContentSection getContentSection(String sSection) throws DataObjectNotFoundException {
        validateURLParameter("content section", sSection);

        Session ses = SessionManager.getSession();
        TransactionContext txn = ses.getTransactionContext();

        //no longer need to explicitly begin txn's
        //txn.beginTxn();

         // The root site node
        SiteNode rootNode = SiteNode.getRootSiteNode();
        BigDecimal rootNodeId = rootNode.getID();

        ContentSection section = null;

        s_log.info("checking for content section " + sSection);
        SiteNode node = SiteNode.getSiteNode("/" + sSection);

        // If the section does not exist yet, create the content section
        if ( rootNodeId.equals(node.getID()) ) {
            section = null;
        } else {
            section = ContentSection.getSectionFromNode(node);
        }

        if (txn.inTxn()) {
            txn.commitTxn();
        }

        return section;
    }

    private void validateURLParameter(String name, String value)
        throws IllegalArgumentException {

        final String pattern = "/[^A-Za-z_0-9\\-]+/";
        Perl5Util util = new Perl5Util();
        if ( util.match(pattern, value) ) {
            throw new IllegalArgumentException
                ("The \"" + name + "\" parameter must contain only " +
                 " alpha-numeric characters, underscores, and/or hyphens.");
        }
    }

    public static void main(String[] args) {
        new Startup().run();

        final Session session = SessionManager.getSession();
        final ParameterLoader loader = new InteractiveParameterLoader(System.in, System.out);

        Script script = new CMSLoader();
        script.load(loader, new ErrorList());
        script.run(new ScriptContext(session, loader));
    }

    private static class ClassArrayParameter extends ClassParameter {
        private static final Logger s_log = Logger.getLogger(ClassArrayParameter.class);

        public ClassArrayParameter(final String name) {
            super(name);
        }

        public Object doRead(ParameterReader paramReader, ErrorList errors) {
            String strParam = paramReader.read(this, errors);
            if (strParam == null) {
                return new Class[] {};
            }
            String[] classParams = StringUtils.split(strParam, ',');

            ArrayList classesList = new ArrayList(classParams.length);
            for (int i=0; i < classParams.length; i++) {
                Class classParam = (Class) super.unmarshal(classParams[i], errors);
                if (s_log.isDebugEnabled()) {
                    s_log.debug("read class " + classParam);
                }
                classesList.add(classParam);
            }

            return (Class[])classesList.toArray(new Class[]{});
        }

        public void doValidate(final Object value, final ErrorList errors) {
            final Class[] classes = (Class[]) value;
            for (int i=0; i < classes.length; i++){
                Class classVal = classes[i];
                super.doValidate(classVal, errors);
            }
            if (classes.length == 0) {
                final ParameterError error = new ParameterError
                    (this, "At least one class must be specified");
                errors.add(error);
            }
        }

        public void doWrite(ParameterWriter writer, final Object value) {
               StringBuffer sbuf = new StringBuffer();
               final Class[] classes = (Class[])value;
               for (int i=0; i < classes.length; i++) {
                   sbuf.append(super.marshal(classes[i]));
                   if (i != classes.length - 1) {
                       sbuf.append(",");
                   }
               }

                writer.write(this, sbuf.toString());
           }
    }
}
