package com.arsdigita.populate.cms;

import com.arsdigita.cms.ContentItem;
import com.arsdigita.cms.ContentSection;
import com.arsdigita.cms.ImageAsset;
import com.arsdigita.cms.ReusableImageAsset;
import com.arsdigita.cms.TextAsset;
import com.arsdigita.cms.workflow.CMSTask;
import com.arsdigita.domain.DomainObjectFactory;
import com.arsdigita.kernel.KernelHelper;
import com.arsdigita.kernel.User;
import com.arsdigita.kernel.UserCollection;
import com.arsdigita.mimetypes.MimeType;
import com.arsdigita.util.Assert;
import com.arsdigita.util.ResourceManager;
import com.arsdigita.workflow.simple.TaskCollection;
import com.arsdigita.workflow.simple.Workflow;
import org.apache.log4j.Logger;

/**
 * Copyright (C) 2001, 2002, 2003 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 * @author bche@redhat.com
 */
public abstract class AbstractCreateContentItem implements CreateContentItem {
    private ImageAsset m_image = null;
    private ReusableImageAsset m_reusableImage = null;

    private static final Logger s_log =
        Logger.getLogger(CreateContentItem.class.getName());

    private static final PopulateConfig s_config = new PopulateConfig();
    static {
        s_config.load();
    }

    /**
     * @see CreateContentItem#setup(String, ContentItem, ContentSection)
     */
    public abstract void setup(String sIdentifier, ContentItem parent,
                               ContentSection section);

    /**
     * @see CreateContentItem#makeContent()
     */
    public abstract void makeContent();

    /**
     * @see CreateContentItem#publish()
     */
    public abstract void publish();

    /**
     * @see CreateContentItem#getContentType()
     */
    public abstract String getContentType();

    /**
     * @see CreateContentItem#getContentTypeName()
     */
    public abstract String getContentTypeName();

    /**
     * @see CreateContentItem#getContentItem()
     */
    public abstract ContentItem getContentItem();


    /**
     * Method makeTextBody.  Returns a string suitable for placing within a text
     * asset
     *
     * @param iIndex an index to place at the beginning of the returned string
     * in order to make it unique
     * @return String
     */
    protected String makeTextBody(int iIndex) {
        StringBuffer sb = new StringBuffer(5000);
        for (int i = 0; i < 20; i++) {
            sb.append(iIndex + " All work and no play makes Jack a dull boy.  ");
        }
        return sb.toString();
    }

    /**
     * Method makeTextBody.  Returns a string suitable for placing within a text
     * asset
     *
     * @return String
     */
    protected String makeTextBody() {
        return makeTextBody(0);
    }


    /**
     * Method createImageAsset.  Creates an Image Asset and returns it
     * @param sName name of the ImageAsset
     * @param parent the ImageAsset's parent
     * @return ImageAsset
     */
    protected ImageAsset createImageAsset(String sName, ContentItem parent) {
        //if we have already created an ImageAsset from the filesystem once,
        //just copy that ImageAsset to create a new one rather than
        //load from the filesystem again
        if (m_image != null) {
            return createImageAsset(m_image, sName, parent);
        }

        s_log.info("  adding image asset to the db");        
        java.io.File imgFile = s_config.getImageFile();
        
        ImageAsset img = new ImageAsset();
        try {
            String sFileName = imgFile.getName();
            if (s_log.isDebugEnabled()) {
                s_log.debug(
                    "adding image file "
                        + sFileName
                        + " with path "
                        + imgFile.getAbsolutePath());
            }
            MimeType mime = MimeType.guessMimeTypeFromFile(sFileName);
            String sMime = "";
            if (mime != null) {
                sMime = mime.getMimeType();
                s_log.debug("setting mime type to " + sMime);
            }
            img.loadFromFile(sFileName, imgFile, sMime);
        } catch (java.io.IOException ioe) {
            s_log.error("IOException loading image " + ioe.getMessage());
        }

        img.setName(sName);
        img.setParent(parent);

        img.save();
        m_image = img;

        return img;
    }

    /**
     * Method createImageAsset.  Creates a new image asset by first copying img
     * and then setting its name and parent.
     * @param img the ImageAsset to copy
     * @param sName name of the new ImageAsset
     * @param parent parent of the new ImageAsset
     * @return ImageAsset
     */
    private ImageAsset createImageAsset(ImageAsset img, String sName,
                                        ContentItem parent) {

        //ImageAsset newImg = (ImageAsset) img.copy();
        ImageAsset newImg = (ImageAsset) ((ImageAsset) DomainObjectFactory.newInstance(img.getOID())).copy(); 
        newImg.setName(sName);
        newImg.setParent(parent);
        newImg.save();

        return newImg;
    }

    /**
     * Method createImageAsset.  Creates an Image Asset and returns it
     * @param sName name of the ImageAsset
     * @param parent the ImageAsset's parent
     * @return ImageAsset
     */
    protected ReusableImageAsset createReusableImageAsset(String sName,
                                                          ContentItem parent) {

        //if we have already created a ReusableImageAsset from the filesystem
        //once, just copy that ReusableImageAsset to create a new one rather
        //than load from the filesystem again
        if (m_reusableImage != null) {
            return createReusableImageAsset(m_reusableImage, sName, parent);
        }

        s_log.info("  adding reusableImage asset to the db");        
        java.io.File imgFile = s_config.getImageFile();

        ReusableImageAsset img = new ReusableImageAsset();
        try {
            String sFileName = imgFile.getName(); 
            if (s_log.isDebugEnabled()) {
                s_log.debug("adding image file " + sFileName + " with path " + imgFile.getAbsolutePath());
            }
            MimeType mime =MimeType.guessMimeTypeFromFile(sFileName);
            String sMime = "";
            if (mime != null) {
                sMime = mime.getMimeType();
                s_log.debug("setting mime type to " + sMime); 
            }            
            img.loadFromFile(sFileName, imgFile, sMime);            
        } catch (java.io.IOException ioe) {
            s_log.error("IOException loading image " + ioe.getMessage());
        }

        img.setName(sName);
        img.setParent(parent);

        img.save();
        m_reusableImage = img;
        
        if (s_log.isDebugEnabled()) {
            s_log.debug("added resuableImage: " + img.getName() + " " + img.getMimeType() + " " + img.getSize());
        }

        return img;
    }

    /**
     * Method createReusableImageAsset.  Creates a new reusableImage asset by
     * first copying img and then setting its name and parent.
     * @param img the ReusableImageAsset to copy
     * @param sName name of the new ReusableImageAsset
     * @param parent parent of the new ReusableImageAsset
     * @return ReusableImageAsset
     */
    private ReusableImageAsset createReusableImageAsset(ReusableImageAsset img,
                                                        String sName,
                                                        ContentItem parent) {
        //ReusableImageAsset newImg = (ReusableImageAsset) img.copy();
        ReusableImageAsset newImg = (ReusableImageAsset) ((ReusableImageAsset) DomainObjectFactory.newInstance(img.getOID())).copy();        
        newImg.setName(sName);
        newImg.setParent(parent);
        newImg.save();

        return newImg;
    }

    /**
     * Method createTextAsset.  Creates and returns a new TextAsset
     * @param sName name of the TextAsset
     * @param sText the text of the TextAsset
     * @param parent the parent of the TextAsset
     * @return TextAsset
     */
    protected TextAsset createTextAsset(String sName, ContentItem parent, String sText) {
        TextAsset txt = new TextAsset();
        txt.setText(sText);
        txt.setName(sName);
        txt.setParent(parent);

        txt.save();
        return txt;
    }

    /**
     * Method createTextAsset.  Creates and returns a new TextAsset with its
     * text automatically set
     *
     * @param sName name of the TextAsset
     * @param parent parent of the TextAsset
     * @return TextAsset
     */
    protected TextAsset createTextAsset(String sName, ContentItem parent) {
        return createTextAsset(sName, parent, makeTextBody());
    }

    /**
     * Method createTextAsset.  Creates and returns a newTextAsset.
     * Automatically sets its text, but will use the integer iIndex to make the
     * text unique.
     *
     * @param sName name of the TextAsset
     * @param parent parent of the TextAsset
     * @param iIndex integer to make the text of the TextAsset unique
     * @return TextAsset
     */
    protected TextAsset createTextAsset(String sName, ContentItem parent,
                                        int iIndex) {

        return createTextAsset(sName, parent, makeTextBody(iIndex));
    }

    /**
     * Method getAdminUser.  Returns a system administrator user
     * @return User
     */
    protected User getAdminUser() {
        UserCollection uc = User.retrieveAll();
        uc.filter(KernelHelper.getSystemAdministratorEmailAddress());
        uc.next();
        User sysadmin = uc.getUser();
        Assert.assertNotNull(sysadmin);
        uc.close();

        return sysadmin;
    }

    /**
     * Method finishTasks.  Starts and finishes all the tasks in Worflow w
     * @param w
     */
    protected void finishTasks(Workflow w) {
        User sysadmin = getAdminUser();

        w.start(sysadmin);

        TaskCollection tasks = w.getTaskCollection();
        try {
            while (tasks.next()) {
                CMSTask task = (CMSTask)tasks.getTask();
                //s_log.info("cmstask " + task.getLabel());
                task.enable();
                try {
                    task.finish(sysadmin, "finished");
                } catch (com.arsdigita.workflow.simple.TaskException e) {
                    s_log.error("TaskException: " + e.getMessage());
                } 
            }
        } finally {
            tasks.close();
        }

    }
}
