/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.ui.folder;


import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.Tree;
import com.arsdigita.bebop.tree.TreeModelBuilder;
import com.arsdigita.bebop.tree.TreeNode;
import com.arsdigita.cms.CMS;
import com.arsdigita.cms.ContentSection;
import com.arsdigita.cms.Folder;
import com.arsdigita.cms.SecurityManager;
import com.arsdigita.kernel.ACSObject;
import com.arsdigita.kernel.Kernel;
import com.arsdigita.kernel.Party;
import com.arsdigita.kernel.User;
import com.arsdigita.kernel.permissions.PermissionManager;
import com.arsdigita.kernel.permissions.PermissionService;
import com.arsdigita.kernel.permissions.PrivilegeDescriptor;
import com.arsdigita.kernel.permissions.UniversalPermissionDescriptor;
import com.arsdigita.kernel.ui.DataQueryTreeIterator;
import com.arsdigita.kernel.ui.DataQueryTreeModel;
import com.arsdigita.kernel.ui.DataQueryTreeNode;
import com.arsdigita.persistence.OID;
import com.arsdigita.util.LockableImpl;

import java.util.Iterator;


/**
 * A {@link com.arsdigita.bebop.tree.TreeModelBuilder} that produces trees
 * containing the folder structure underneath a root folder. The root
 * folder can be changed by overriding {@link #getRoot getRoot}.
 *
 * @author <a href="mailto:tri@arsdigita.com">Tri Tran</a>
 * @author <a href="mailto:lutter@arsdigita.com">David Lutterkort</a>
 *
 * @version $Id: //cms/dev/src/com/arsdigita/cms/ui/folder/FolderTreeModelBuilder.java#11 $
 */

public class FolderTreeModelBuilder extends LockableImpl
    implements TreeModelBuilder {

    public static final String versionId = "$Id: //cms/dev/src/com/arsdigita/cms/ui/folder/FolderTreeModelBuilder.java#11 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private PrivilegeDescriptor TREE_DESCRIPTOR = 
        SecurityManager.CMS_PREVIEW_ITEM_DESCRIPTOR;
    /**
     * Make a tree model that lists the hierarchy of folders underneath the
     * folder returnedby {@link #getRoot getRoot}.
     *
     * @param t the tree in which the model is used
     * @param s represents the current request
     * @return a tree model that lists the hierarchy of folders underneath the
     * folder returnedby {@link #getRoot getRoot}.
     */
    public com.arsdigita.bebop.tree.TreeModel makeModel(Tree t, PageState s) {
        return new DataQueryTreeModel(getRoot(s).getID(),
                                      "com.arsdigita.cms.getRootFolder",
                                      "com.arsdigita.cms.getSubFolders") {
                protected DataQueryTreeIterator getDataQueryTreeIterator
                    (DataQueryTreeNode node, String getSubCategories) {
                    DataQueryTreeIterator iterator =
                        new DataQueryTreeIterator(node, getSubCategories);
                    iterator.filterQuery
                        (ACSObject.ID, 
                         TREE_DESCRIPTOR,
                         Kernel.getContext().getParty().getOID());
                    return iterator;
                }

                public Iterator getChildren(TreeNode node, PageState data) {
                    Party party = Kernel.getContext().getParty();
                    OID partyOID = null;
                    if (party == null) {
                        partyOID = new OID(User.BASE_DATA_OBJECT_TYPE, 
                                           PermissionManager.VIRTUAL_PUBLIC_ID);
                    } else {
                        partyOID = party.getOID();
                    }
                    UniversalPermissionDescriptor universalPermission =
                        new UniversalPermissionDescriptor
                        (SecurityManager.CMS_PREVIEW_ITEM_DESCRIPTOR, partyOID);
                    if (PermissionService.checkPermission
                        (universalPermission)) {
                        // the person is an admin so we just pass in the
                        // standard, non filtered query
                        return getDataQueryTreeIterator
                            ((DataQueryTreeNode)node, 
                             "com.arsdigita.cms.getSubFolders");
                    } else {
                        DataQueryTreeIterator iterator = 
                            getDataQueryTreeIterator
                            ((DataQueryTreeNode)node, 
                             "com.arsdigita.cms.getSubFoldersWithPermissions");
                        // now we need to set the parameters
                        iterator.setParameter
                            ("party", partyOID.get("id"));
                        iterator.setParameter
                            ("priv", TREE_DESCRIPTOR.getName());
                        return iterator;
                    }
                }
            };
    }

    /**
     * Retrn the root folder for the tree model in the current
     * request.
     * @param s represents the current request
     * @return the root folder for the tree
     * @post return != null
     */
    protected Folder getRoot(PageState s)
        throws IllegalStateException {
        ContentSection sec = CMS.getContext().getContentSection();
        return sec.getRootFolder();
    }


}
