/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.ui.category;

import com.arsdigita.bebop.FormProcessException;
import com.arsdigita.bebop.GridPanel;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.SaveCancelSection;
import com.arsdigita.bebop.event.ParameterEvent;
import com.arsdigita.bebop.event.ParameterListener;
import com.arsdigita.bebop.form.FormErrorDisplay;
import com.arsdigita.bebop.form.TextArea;
import com.arsdigita.bebop.form.TextField;
import com.arsdigita.bebop.parameters.NotNullValidationListener;
import com.arsdigita.bebop.parameters.TrimmedStringParameter;
import com.arsdigita.categorization.Category;
import com.arsdigita.categorization.CategoryCollection;
import com.arsdigita.cms.ui.CMSForm;
import com.arsdigita.cms.util.GlobalizationUtil;
import com.arsdigita.globalization.GlobalizedMessage;

import org.apache.log4j.Logger;

/**
 * A form which creates a new category. Extends the edit form for
 * convenience.
 *
 * @author Michael Pih
 * @author Stanislav Freidin &lt;sfreidin@redhat.com&gt;
 * @version $Id: //cms/dev/src/com/arsdigita/cms/ui/category/CategoryBaseForm.java#10 $
 */
class CategoryBaseForm extends CMSForm {
    private static final Logger s_log =
        Logger.getLogger(CategoryBaseForm.class);

    final CategoryRequestLocal m_parent;

    final FormErrorDisplay m_errors;
    final TextField m_title;
    final TextArea m_description;
    final SaveCancelSection m_saveCancelSection;

    /**
     * Constructor.
     */
    CategoryBaseForm(final String string, final CategoryRequestLocal parent) {
        super("AddSubcategories");

        m_parent = parent;

        // Form header
        Label header = new Label(gz("cms.ui.category.add"));
        add(header, GridPanel.FULL_WIDTH);

        // Form errors
        m_errors = new FormErrorDisplay(this);
        add(m_errors, GridPanel.FULL_WIDTH);

        // Name
        m_title = new TextField(new TrimmedStringParameter("name"));
        m_title.setSize(30);
        m_title.setMaxLength(30);
        m_title.addValidationListener(new NotNullValidationListener());
        m_title.addValidationListener(new TitleUniqueListener());
        add(new Label(gz("cms.ui.name")));
        add(m_title);

        // Description
        m_description = new TextArea(new TrimmedStringParameter("description"));
        m_description.setWrap(TextArea.SOFT);
        m_description.setRows(5);
        m_description.setCols(40);
        m_description.addValidationListener(new NotNullValidationListener());
        add(new Label(gz("cms.ui.description")));
        add(m_description);

        // Save and cancel buttons
        m_saveCancelSection = new SaveCancelSection();
        add(m_saveCancelSection, GridPanel.FULL_WIDTH | GridPanel.LEFT);
    }

    public final boolean isCancelled(final PageState state) {
        return m_saveCancelSection.getCancelButton().isSelected(state);
    }

    static GlobalizedMessage gz(final String key) {
        return GlobalizationUtil.globalize(key);
    }

    static String lz(final String key) {
        return (String) gz(key).localize();
    }

    class TitleUniqueListener implements ParameterListener {
        public final void validate(final ParameterEvent e)
            throws FormProcessException {

            final PageState state = e.getPageState();
            final String title = (String) m_title.getValue(state);

            final Category parent = m_parent.getCategory(state);
            final CategoryCollection children = parent.getChildren();

            while (children.next()) {
                final Category child = children.getCategory();

                if (child.getName().equalsIgnoreCase(title)) {
                    throw new FormProcessException
                        (lz("cms.ui.category.name_not_unique"));
                }
            }
        }
    }
}
