/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.ui.authoring;

import com.arsdigita.bebop.Page;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.event.ActionEvent;
import com.arsdigita.bebop.event.ActionListener;
import com.arsdigita.bebop.event.RequestEvent;
import com.arsdigita.bebop.event.RequestListener;
import com.arsdigita.bebop.parameters.StringParameter;
import com.arsdigita.cms.ContentItem;
import com.arsdigita.cms.ItemSelectionModel;
import com.arsdigita.cms.dispatcher.Utilities;
import com.arsdigita.cms.ui.ContentItemPage;
import com.arsdigita.cms.ui.SecurityPropertyEditor;
import com.arsdigita.toolbox.ui.ComponentAccess;


import org.apache.log4j.Logger;

/**
 * A simple implementation of an Authoring Kit editing step.
 * Extends {@link SecurityPropertyEditor} and provides authoring kit
 * integration. See the authoring kit documentation for more
 * info.
 * <p>
 *
 * Child classes should
 * a). call setDisplayComponent()
 * b). call add() zero or more times
 *
 * @author Stanislav Freidin
 */
public class SimpleEditStep extends SecurityPropertyEditor 
    implements AuthoringStepComponent, RequestListener {

    private static final Logger s_log =
        Logger.getLogger( SimpleEditStep.class );

    public static final String versionId = "$Id: //cms/dev/src/com/arsdigita/cms/ui/authoring/SimpleEditStep.java#10 $ by $Author: mbooth $, $DateTime: 2004/05/06 13:34:51 $";

    private AuthoringKitWizard m_parent;
    private ItemSelectionModel m_itemModel;
    private String m_defaultEditKey = null;

    private StringParameter m_streamlinedCreationParam;
    private static final String STREAMLINED = "_streamlined";
    private static final String STREAMLINED_DONE = "1";

    /**
     * Construct a new SimpleEditStep component
     *
     * @param itemModel The {@link ItemSelectionModel} which will
     *   be responsible for loading the current item
     *
     * @param parent The parent wizard which contains the form.
     *   The component may use the wizard's methods, such as stepForward
     *   and stepBack, in its process listener.
     */
    public SimpleEditStep(ItemSelectionModel itemModel, AuthoringKitWizard parent) {
        this(itemModel, parent, "");
    }
    /**
     * Construct a new SimpleEditStep component
     *
     * @param itemModel The {@link ItemSelectionModel} which will
     *   be responsible for loading the current item
     *
     * @param parent The parent wizard which contains the form.
     *   The component may use the wizard's methods, such as stepForward
     *   and stepBack, in its process listener.
     *
     * @param paramSuffix Additional global parameter name suffix if
     * there are multiple SimpleEditStep instances in an authoring kit.
     */
    public SimpleEditStep(ItemSelectionModel itemModel, AuthoringKitWizard parent, String paramSuffix) {
        super();
        m_parent = parent;
        m_itemModel = itemModel;

        m_streamlinedCreationParam = 
            new StringParameter(parent.getContentType().getAssociatedObjectType() + "_properties_done" + paramSuffix);
        
            
        parent.getList().addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent e) {
                    PageState state = e.getPageState();
                    showDisplayPane(state);
                }
            });
    }

    /** 
     * Registers globa state param for cancelling streamlined
     * creation
     */
    public void register(Page p) {
        super.register(p);
        p.addGlobalStateParam(m_streamlinedCreationParam);
        p.addRequestListener(this);
    }

    /**
     * @return the parent wizard
     */
    public AuthoringKitWizard getParentWizard() {
        return m_parent;
    }

    /**
     * @return The item selection model
     */
    public ItemSelectionModel getItemSelectionModel() {
        return m_itemModel;
    }

    /**
     * Forward to the next step if the streamlined creation parameter
     * is turned on _and_  the streamlined_creation global state param
     * is set to 'active'
     *
     * @param state the PageState
     */
    public void maybeForwardToNextStep(PageState state) {
        if (ContentItemPage.isStreamlinedCreationActive(state) &&
            !STREAMLINED_DONE.equals(state.getValue(m_streamlinedCreationParam))) {
            state.setValue(m_streamlinedCreationParam, STREAMLINED_DONE);
            fireCompletionEvent(state);
        }
    }

    /**
     * Cancel streamlined creation for this step if the streamlined
     * creation parameter is turned on _and_ the streamlined_creation
     * global state param is set to 'active'
     *
     * @param state the PageState
     */
    public void cancelStreamlinedCreation(PageState state) {
        if (ContentItemPage.isStreamlinedCreationActive(state)) {
            state.setValue(m_streamlinedCreationParam, STREAMLINED_DONE);
        }
    }
    
    
    public void setDefaultEditKey(String key) {
        m_defaultEditKey = key;
    }

    /**
     * Open the edit component if the streamlined
     * creation parameter is turned on _and_ the streamlined_creation
     * global state param is set to 'active'
     *
     * @param state the PageState
     */
    public void pageRequested(RequestEvent e) {
        PageState state = e.getPageState();

        // XXX: This method is called on every page request for every authoring
        // step in every authoring kit. This has in the past revealed a caching
        // side-effect bug, but should in the main be harmless. Except of course
        // for performance.
        // Ideally this method would only be called for a single authoring step
        // on each page load. However, at the stage that this is called,
        // visibility has not been set, and getting the selected authoring kit
        // or component is not straightforward, and would almost certainly
        // involve duplicating code.
        // This need to be rethought.
        //if( !state.isVisibleOnPage( this ) ) return;

        if (m_defaultEditKey != null && m_itemModel.getSelectedItem(state) != null) {
            ComponentAccess ca = (ComponentAccess) getAccessMap().get(m_defaultEditKey);
            
            if (ContentItemPage.isStreamlinedCreationActive(state) &&
                !STREAMLINED_DONE.equals(state.getValue(m_streamlinedCreationParam)) &&
                ca != null && ca.canAccess(state, Utilities.getSecurityManager(state))) {
                showComponent(state, m_defaultEditKey);
            }
        }
        
    }
}
