/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.ui.authoring;


import com.arsdigita.bebop.BoxPanel;
import com.arsdigita.bebop.Form;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.event.PrintEvent;
import com.arsdigita.bebop.event.PrintListener;
import com.arsdigita.bebop.form.Option;
import com.arsdigita.bebop.form.OptionGroup;
import com.arsdigita.bebop.form.SingleSelect;
import com.arsdigita.bebop.form.Submit;
import com.arsdigita.bebop.parameters.BigDecimalParameter;
import com.arsdigita.cms.ContentSection;
import com.arsdigita.cms.ContentType;
import com.arsdigita.cms.ContentTypeCollection;
import com.arsdigita.globalization.GlobalizedMessage;
import com.arsdigita.util.UncheckedWrapperException;
import com.arsdigita.xml.Element;

import java.math.BigDecimal;

/**
 * A form which displays a select box of all content types available
 * under the given content section, and forwards to the item creation
 * UI when the user selects a content type to instantiate.
 *
 * @author Stanislav Freidin (sfreidin@arsdigtia.com)
 * @version $Revision: #11 $ $DateTime: 2004/04/07 16:07:11 $
 */
public abstract class NewItemForm extends Form {

    public static final String versionId = "$Id: //cms/dev/src/com/arsdigita/cms/ui/authoring/NewItemForm.java#11 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    public static final String RESOURCE_BUNDLE =
        "com.arsdigita.cms.ui.authoring.AuthoringResources";

    public static String DP_TYPE_PREFIX =
        "com.arsdigita.dp.";

    private SingleSelect m_typeWidget;
    private Submit m_submit;
    private Label m_emptyLabel;
    private Label m_createLabel;

    public static final String TYPE_ID = "tid";

    /**
     * Construct a new NewItemForm
     *
     */
    public NewItemForm(String name) {
        super(name, new BoxPanel(BoxPanel.HORIZONTAL));
        setIdAttr("new_item_form");

        BoxPanel panel = (BoxPanel) getPanel();
        panel.setWidth("2%");
        panel.setBorder(0);

        m_emptyLabel = new Label
            (globalize("cms.ui.authoring.no_types_registered"), false);
        m_emptyLabel.setIdAttr("empty_label");
        add(m_emptyLabel);

        m_createLabel = new Label
            (globalize("cms.ui.authoring.create_new"), false);
        m_createLabel.setIdAttr("create_label");
        add(m_createLabel);

        m_typeWidget = new SingleSelect(new BigDecimalParameter(TYPE_ID));
        try {
            m_typeWidget.addPrintListener(new PrintListener() {

                    // Read the content section's content types and add them as options
                    public void prepare(PrintEvent e) {
                        OptionGroup o = (OptionGroup)e.getTarget();
                        PageState state = e.getPageState();

                        ContentSection section = getContentSection(state);

                        ContentTypeCollection c = section.getCreatableContentTypes();

                        c.addOrder(ContentType.LABEL);

                        if(!c.isEmpty()) {
                            // Add content types
                            while(c.next()) {
                                ContentType type = c.getContentType();
                                //for dp content type label localization
                                //String t = type.getAssociatedObjectType();
                                String cn = type.getClassName();
                                String l = type.getLabel();
                                if (cn.startsWith(DP_TYPE_PREFIX, 0)) {
                                    o.addOption(new Option
                                                (type.getID().toString(),
                                                 new Label(globalize(l.replace(' ','_')))));
                                } else {
                                    o.addOption(new Option
                                                (type.getID().toString(), type.getLabel()));
                                }
                            }
                            c.reset();
                        }
                    }
                });
        } catch (java.util.TooManyListenersException e) {
            throw new UncheckedWrapperException("Too  many listeners: " + e.getMessage(), e);
        }

        add(m_typeWidget);

        m_submit = new Submit
            ("new", globalize("cms.ui.authoring.go"));
        add(m_submit);

    }

    public abstract ContentSection getContentSection(PageState state);

    public BigDecimal getTypeID(PageState state) {
        return (BigDecimal) m_typeWidget.getValue(state);
    }

    public final SingleSelect getTypeSelect() {
        return m_typeWidget;
    }

    // Generate XML - show/hide labels/widgets
    public void generateXML(PageState state, Element parent) {

        if ( isVisible(state) ) {
            ContentSection section = getContentSection(state);

            ContentTypeCollection c = section.getCreatableContentTypes();
            boolean isEmpty = c.isEmpty();
            c.close();

            m_createLabel.setVisible(state, !isEmpty);
            m_typeWidget.setVisible(state, !isEmpty);
            m_submit.setVisible(state, !isEmpty);
            m_emptyLabel.setVisible(state, isEmpty);

            super.generateXML(state, parent);
        }
    }

    /**
     * Getting the GlobalizedMessage using a CMS Class targetBundle.
     *
     * @param key The resource key
     * @pre ( key != null )
     */
    public static GlobalizedMessage globalize(String key) {
        return new GlobalizedMessage(key, RESOURCE_BUNDLE);
    }

}
