/*
 * Copyright (C) 2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.ui.authoring;

import com.arsdigita.bebop.form.Widget;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.parameters.ArrayParameter;
import com.arsdigita.bebop.parameters.StringParameter;
import com.arsdigita.bebop.parameters.BigDecimalParameter;
import com.arsdigita.xml.Element;
import com.arsdigita.xml.XML;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.OID;
import com.arsdigita.domain.DomainObjectFactory;
import com.arsdigita.categorization.Category;
import com.arsdigita.categorization.CategoryCollection;
import com.arsdigita.cms.CMS;
import java.util.Set;
import java.util.HashSet;
import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.math.BigDecimal;

public class CategoryWidget extends Widget {
    
    private BigDecimalParameter m_root;
    private StringParameter m_mode;

    public CategoryWidget(String name,
                          BigDecimalParameter root,
                          StringParameter mode) {
        super(new ArrayParameter(new BigDecimalParameter(name)));
        
        m_root = root;
        m_mode = mode;
    }

    protected String getType() {
        return "category";
    }
    public boolean isCompound() {
        return false;
    }
    
    protected void generateWidget(PageState state, 
                                  Element parent) {
        Element widget = parent.newChildElement("cms:categoryWidget",
                                                CMS.CMS_XML_NS);
        exportAttributes(widget);

        widget.addAttribute("mode", (String)state.getValue(m_mode));
        widget.addAttribute("name", getName());

        Set ids = new HashSet();

        BigDecimal[] values = (BigDecimal[])getValue(state);
        if (values != null) {
            for (int i = 0 ; i < values.length ; i++) {
                ids.add(values[i]);
            }
        }
        
        Category root = (Category)DomainObjectFactory.newInstance(
            new OID(Category.BASE_DATA_OBJECT_TYPE,
                    (BigDecimal)state.getValue(m_root))
        );

        CategoryCollection cats = root.getDescendants();
        cats.addEqualsFilter("parents.link.relationType", "child");
        cats.addPath("parents.link.sortKey");
        cats.addPath("parents.id");
        
        Map children = new HashMap();
        while (cats.next()) {
            Category cat = cats.getCategory();
            BigDecimal parentID = (BigDecimal)cats.get("parents.id");
            
            List childList = (List)children.get(parentID);
            if (childList == null) {
                childList = new ArrayList();
                children.put(parentID, childList);
            }
            
            childList.add(new CategorySortKeyPair
                          (cat,(BigDecimal)cats.get("parents.link.sortKey")));
        }
        
        generateCategory(widget, null, root, null, ids, children);
    }

    public void generateCategory(Element parent,
                                 String path,
                                 Category cat,
                                 BigDecimal sortKey,
                                 Set selected,
                                 Map children) {
        Element el = new Element("cms:category",
                                 CMS.CMS_XML_NS);
        
        el.addAttribute("id", XML.format(cat.getID()));
        el.addAttribute("name", cat.getName());
        el.addAttribute("description", cat.getDescription());
        el.addAttribute("isSelected", selected.contains(cat.getID()) ? "1" : "0");
        el.addAttribute("isAbstract", cat.isAbstract() ? "1" : "0");
        el.addAttribute("isEnabled", cat.isEnabled() ? "1" : "0");
        if (sortKey != null) {
            el.addAttribute("sortKey", sortKey.toString());
        }
        
        String fullname = path == null ? "/" : path + " > " + cat.getName();
        el.addAttribute("fullname", fullname);
        
        parent.addContent(el);

        List c = (List)children.get(cat.getID());
        if (c != null) {
            Iterator i = c.iterator();
            while (i.hasNext()) {
                CategorySortKeyPair pair = (CategorySortKeyPair) i.next();
                Category child = pair.getCategory();
                BigDecimal childSortKey = pair.getSortKey();
                generateCategory(el, fullname, child, 
                                 childSortKey, selected, children);
            }
        }        
    }
    private class CategorySortKeyPair {
        private Category m_category;
        private BigDecimal m_sortKey;

        public CategorySortKeyPair(Category category, BigDecimal sortKey) {
            m_category = category;
            m_sortKey = sortKey;
        }
        public Category getCategory() {
            return m_category;
        }
        public BigDecimal getSortKey() {
            return m_sortKey;
        }
    }
}
