/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.ui;

import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.ParameterSingleSelectionModel;
import com.arsdigita.bebop.SingleSelectionModel;
import com.arsdigita.bebop.parameters.BigDecimalParameter;
import com.arsdigita.cms.Article;
import com.arsdigita.cms.ImageAsset;
import com.arsdigita.cms.ImageAssetCollection;
import com.arsdigita.cms.ItemSelectionModel;
import com.arsdigita.domain.DomainObject;


/**
 * Selects a single image for an article.
 *
 * @version $IdS
 * @author <a href="mailto:sfreidin@arsdigita.com">Stanislav Freidin</a>
 *
 * @see com.arsdigita.cms.ui.authoring.ArticleImage
 * @see com.arsdigita.cms.ui.ArticleImageDisplay
 */

public class SingleImageSelectionModel extends ItemSelectionModel {

    public static final String versionId = "$Id: //cms/dev/src/com/arsdigita/cms/ui/SingleImageSelectionModel.java#8 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private ItemSelectionModel m_articleModel;

    /**
     * Construct a new <code>SingleImageSelectionModel</code>
     *
     * @param javaClass the Java class name of the {@link ImageAsset} subclass
     *   that this model deals with
     * @param objetcType the PDL object type of the {@link ImageAsset} subclass
     *   that this model deals with
     * @param param the {@link BigDecimalParameter} where the image ID will
     *   be stored
     * @param articleModel the {@link ItemSelectionModel} which will supply
     *   the current article
     */
    public SingleImageSelectionModel(
                                     String javaClass, String objectType, BigDecimalParameter param,
                                     ItemSelectionModel articleModel
                                     ) {
        this(javaClass, objectType, new ParameterSingleSelectionModel(param),
             articleModel);
    }

    /**
     * Construct a new <code>SingleImageSelectionModel</code>
     *
     * @param javaClass the Java class name of the {@link ImageAsset} subclass
     *   that this model deals with
     * @param objetcType the PDL object type of the {@link ImageAsset} subclass
     *   that this model deals with
     * @param imageModel the {@link SingleSelectionModel} which will store the
     *   image ID
     * @param articleModel the {@link ItemSelectionModel} which will supply
     *   the current article
     */
    public SingleImageSelectionModel(
                                     String javaClass, String objectType, SingleSelectionModel imageModel,
                                     ItemSelectionModel articleModel
                                     ) {
        super(javaClass, objectType, imageModel);
        m_articleModel = articleModel;
    }

    /**
     * Construct a new <code>SingleImageSelectionModel</code>
     *
     * @param imageModel the {@link SingleSelectionModel} which will store the
     *   image ID
     * @param articleModel the {@link ItemSelectionModel} which will supply
     *   the current article
     */
    public SingleImageSelectionModel(
                                     SingleSelectionModel imageModel, ItemSelectionModel articleModel
                                     ) {
        this(ImageAsset.class.getName(), ImageAsset.BASE_DATA_OBJECT_TYPE,
             imageModel, articleModel);
    }

    /**
     * Construct a new <code>SingleImageSelectionModel</code>
     *
     * @param param the {@link BigDecimalParameter} where the image ID will
     *   be stored
     *
     * @param articleModel the {@link ItemSelectionModel} which will supply
     *   the current article
     */
    public SingleImageSelectionModel(
                                     BigDecimalParameter param, ItemSelectionModel articleModel
                                     ) {
        this(ImageAsset.class.getName(), ImageAsset.BASE_DATA_OBJECT_TYPE,
             param, articleModel);
    }

    // Load the first asset for the article, if neccessary.
    private void checkAsset(PageState state) {
        if ( !isInitialized(state) ) {
            // Load the object from the item.
            com.arsdigita.cms.ContentItem temp =
                (com.arsdigita.cms.ContentItem)m_articleModel.getSelectedObject(state);
            Article item = null;
            if ( temp != null ) {
                item =
                    (Article) com.arsdigita.cms.ACSObjectFactory.castContentItem(temp);
            }
            if ( item != null ) {
                ImageAssetCollection images = item.getImages();
                if ( images.next() ) {
                    setSelectedObject(state, images.getImage());
                    images.close();
                }
            }
        }
    }

    /**
     * Get the currently selected image. If no image is selected,
     * select the first image for the article.
     */
    public DomainObject getSelectedObject(PageState state) {
        checkAsset(state);
        return super.getSelectedObject(state);
    }

    /**
     * Get the id of the currently selected image
     */
    public Object getSelectedKey(PageState state) {
        checkAsset(state);
        return super.getSelectedKey(state);
    }

    /**
     * @return the {@link ItemSelectionModel} which supplies the article
     */
    public ItemSelectionModel getArticleSelectionModel() {
        return m_articleModel;
    }
}
