/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.ui;


import com.arsdigita.bebop.BoxPanel;
import com.arsdigita.bebop.Form;
import com.arsdigita.bebop.FormData;
import com.arsdigita.bebop.FormProcessException;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.Page;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.Paginator;
import com.arsdigita.bebop.ParameterSingleSelectionModel;
import com.arsdigita.bebop.SaveCancelSection;
import com.arsdigita.bebop.SingleSelectionModel;
import com.arsdigita.bebop.event.FormInitListener;
import com.arsdigita.bebop.event.FormProcessListener;
import com.arsdigita.bebop.event.FormSectionEvent;
import com.arsdigita.bebop.event.FormSubmissionListener;
import com.arsdigita.bebop.event.TableActionListener;
import com.arsdigita.bebop.form.Submit;
import com.arsdigita.bebop.form.TextField;
import com.arsdigita.bebop.parameters.NotNullValidationListener;
import com.arsdigita.bebop.parameters.StringParameter;
import com.arsdigita.cms.ContentItem;
import com.arsdigita.cms.util.GlobalizationUtil;


/**
 * Sticks a form at the top of an {@link ImageBrowser} in order to
 * search images by keyword
 *
 * @author Stanislav Freidin (sfreidin@arsdigita.com)
 * @version $Id: //cms/dev/src/com/arsdigita/cms/ui/ImageChooser.java#9 $
 */
public class ImageChooser extends BoxPanel {

    public static final String versionId = "$Id: //cms/dev/src/com/arsdigita/cms/ui/ImageChooser.java#9 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private ImageKeywordForm m_form;
    private ImageBrowser m_browser;
    private Paginator m_paginator;
    private StringParameter m_keyword;
    private SingleSelectionModel m_sel;

    public static final String KEYWORD = "kw";
    public static int LIST_SIZE = 20;


    /**
     * Construct a new ImageChooser
     *
     * @param context the context for the retrieved items. Should be
     *   {@link ContentItem#DRAFT} or {@link ContentItem#LIVE}
     */
    public ImageChooser(String context) {
        super(BoxPanel.VERTICAL);

        m_keyword = new StringParameter(KEYWORD);
        m_sel = new ParameterSingleSelectionModel(m_keyword);
        m_form = new ImageKeywordForm(m_sel);
	DefaultImageBrowserModelBuilder modelBuilder = 
	    new DefaultImageBrowserModelBuilder(m_sel, context);
        m_browser = new ImageBrowser(modelBuilder);
	modelBuilder.setImageBrowser(m_browser);


	m_paginator = new Paginator
	    (modelBuilder,
	     LIST_SIZE);
        super.add(m_form);
	super.add(m_paginator);
        super.add(m_browser);
    }

    /**
     * Construct a new ImageChooser
     */
    public ImageChooser() {
        this(ContentItem.DRAFT);
    }

    /**
     * Add the "keyword" parameter to the page state
     */
    public void register(Page p) {
        p.addComponent(this);
        p.addComponentStateParam(this, m_keyword);
    }

    /**
     * Add an action listener to the browser. The inner class
     * {@link ImageBrowser.LinkActionListener} will probably be
     * used here.
     *
     * @param l the action listener.
     */
    public void addImageActionListener(TableActionListener l) {
        m_browser.addTableActionListener(l);
    }

    /**
     * Add a submission listener to the form. The listener will
     * fire whenever a button on the form is clicked.
     *
     * @param l the action listener.
     */
    public void addFormSubmissionListener(FormSubmissionListener l) {
        m_form.addSubmissionListener(l);
    }

    /**
     * @return the "Cancel" button in the form
     */
    public Submit getFormCancelButton() {
        return m_form.getSaveCancelSection().getCancelButton();
    }

    /**
     * @return the keyword selection model
     */
    public SingleSelectionModel getKeywordModel() {
        return m_sel;
    }

    /**
     * Set the specified keyword. All images matching the keyword
     * will be displayed in the browser.
     *
     * @param state The page state
     * @param word The new keyword
     */
    public void setKeyword(PageState state, String word) {
        m_sel.setSelectedKey(state, word);
    }

    /**
     * Clear the selection in the browser
     *
     * @param state The page state
     */
    public void clearSelection(PageState state) {
        m_browser.clearSelection(state);
    }

    /**
     * @param state The page state
     * @return the current keyword
     */
    public String getKeyword(PageState state) {
        return (String)m_sel.getSelectedKey(state);
    }

    /**
     * @return the form
     */
    public ImageKeywordForm getForm() {
        return m_form;
    }

    /**
     * @return the image browser
     */
    public ImageBrowser getImageBrowser() {
        return m_browser;
    }

    /**
     * Clear the keyword used in the keyword form
     *
     * @param s The page state
     */
    public void clearKeyword(PageState s) {
        m_sel.clearSelection(s);
    }

    /**
     * The form which specifies a keyword for the image browser.
     */
    public static class ImageKeywordForm extends Form
        implements FormProcessListener, FormInitListener {

        private SingleSelectionModel m_sel;
        private SaveCancelSection m_saveCancel;

        public static String WORD = "word";

        /**
         * Construct a new ImageKeywordForm
         * @param sel The SingleSelectionModel which the form will use to
         *   set the keyword
         */
        public ImageKeywordForm(SingleSelectionModel sel) {
            super("ImageKeywordForm", new BoxPanel(BoxPanel.HORIZONTAL));
            m_sel = sel;

            add(new Label(GlobalizationUtil.globalize("cms.ui.enter_a_keyword")));
            TextField t = new TextField(WORD);
	    // allow null keyword field for view all
            //t.addValidationListener(new NotNullValidationListener("keyword"));
            add(t);
            m_saveCancel = new SaveCancelSection();
            m_saveCancel.getSaveButton().setButtonLabel("Search");
            add(m_saveCancel);

            addProcessListener(this);
            addInitListener(this);

            setClassAttr("imageKeywordForm");
        }

        /**
         * Set the keyword in the text widget
         */
        public void init(FormSectionEvent e) throws FormProcessException {
            PageState s = e.getPageState();
            FormData data = e.getFormData();
            data.put(WORD, m_sel.getSelectedKey(s));
        }

        /**
         * @return the SaveCancelSection
         */
        public SaveCancelSection getSaveCancelSection() {
            return m_saveCancel;
        }

        /**
         * @return the SingleSelectionModel
         */
        public SingleSelectionModel getKeywordModel() {
            return m_sel;
        }

        public void process(FormSectionEvent e) {
            FormData data = e.getFormData();
            PageState state = e.getPageState();
            m_sel.setSelectedKey(state, (String)data.get(WORD));
        }
    }
}
