/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.search;

import com.arsdigita.cms.ContentItem;
import com.arsdigita.cms.ContentPage;

import com.arsdigita.domain.DomainObject;
import com.arsdigita.domain.DomainObjectTextRenderer;
import com.arsdigita.domain.DomainObjectXMLRenderer;

import com.arsdigita.kernel.Party;

import com.arsdigita.search.MetadataProvider;
import com.arsdigita.search.ContentProvider;
import com.arsdigita.search.ContentType;

import com.arsdigita.persistence.metadata.ObjectType;
import com.arsdigita.util.StringUtils;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.ArrayList;

import org.apache.log4j.Logger;

/**
 * This class is an implementation of the Search metadata provider that uses the
 * <code>DomainObjectTextRenderer</code> to extract search content for any
 * subclass of {@link com.arsdigita.cms.ContentPage}.
 *
 * @author <a href="mailto:berrange@redhat.com">Daniel Berrange</a>
 * @version $Revision: #4 $ $Date: 2004/04/07 $
 */
public class ContentPageMetadataProvider implements MetadataProvider {

    public static final String ADAPTER_CONTEXT =
        ContentPageMetadataProvider.class.getName();
    
    private static final Logger s_log = 
        Logger.getLogger(ContentPageMetadataProvider.class);

    public final static String versionId =
        "$Id: //cms/dev/src/com/arsdigita/cms/search/ContentPageMetadataProvider.java#4 $"
        + " by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    public String getTypeSpecificInfo(DomainObject dobj) {
        ContentPage item = (ContentPage) dobj;
        return ContentItem.LIVE.equals(item.getVersion()) ?
            ContentItem.LIVE : ContentItem.DRAFT;
    }

    public java.util.Locale getLocale(DomainObject dobj) {
        ContentPage item = (ContentPage)dobj;
        String lang = item.getLanguage();
        return lang == null ? null : new java.util.Locale(lang,"");
    }

    public String getTitle(DomainObject dobj) {
        ContentPage item = (ContentPage)dobj;
        String title = item.getTitle();
        if (StringUtils.emptyString(title)) {
            throw new IllegalArgumentException(
                "ContentPage must have non-blank title!"
            );
        }
        return title;
    }

    public String getSummary(DomainObject dobj) {
        ContentPage item = (ContentPage)dobj;
        return item.getSearchSummary();
    }
    
    public ContentProvider[] getContent(DomainObject dobj,
                                        ContentType type) {
        List content = new ArrayList();
        
        if (type == ContentType.XML) {
            if (s_log.isDebugEnabled()) {
                s_log.debug("Adding XML Content");
            }
            content.add(new ContentPageXMLContentProvider("xml",
                                                          (ContentPage)dobj));
        } else if (type == ContentType.TEXT) {
            if (s_log.isDebugEnabled()) {
                s_log.debug("Adding Text Content");
            }
            content.add(new ContentPageTextContentProvider("text",
                                                           (ContentPage)dobj));
        } else if (type == ContentType.RAW) {        
            if (s_log.isDebugEnabled()) {
                s_log.debug("Adding Raw Content");
            }
            ContentPageAssetExtractor ex = new ContentPageAssetExtractor();
            ex.walk(dobj, ContentPageAssetExtractor.class.getName());
            content.addAll(ex.getContent());
        }

        return (ContentProvider[])content.toArray(
            new ContentProvider[content.size()]);
    }

    public Date getCreationDate(DomainObject dobj) {
        ContentPage item = (ContentPage)dobj;
        return item.getCreationDate();
    }

    public Party getCreationParty(DomainObject dobj) {
        ContentPage item = (ContentPage)dobj;
        return item.getCreationUser();
    }

    public Date getLastModifiedDate(DomainObject dobj) {
        ContentPage item = (ContentPage)dobj;
        return item.getLastModifiedDate();
    }

    public Party getLastModifiedParty(DomainObject dobj) {
        ContentPage item = (ContentPage)dobj;
        return item.getLastModifiedUser();
    }
}
