/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms;

import java.util.StringTokenizer;

/**
 * Contains utility methods which are useful for constructing
 * search queries
 *
 * @author Stanislav Freidin (sfreidin@arsdigita.com)
 * @version $Id: //cms/dev/src/com/arsdigita/cms/SearchFormatter.java#5 $
 */

public class SearchFormatter {

    public static final String versionId = "$Id: //cms/dev/src/com/arsdigita/cms/SearchFormatter.java#5 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    /**
     * A list of all the special characters which will
     * be filtered out
     */
    public static final String SPECIAL_CHARS =
        "|&-*;{}%_$?!()\':@.<>#^+=[]~`\", \t\r\n";

    /**
     * A list of common words which will be filtered out
     */
    public static final String[] COMMON_WORDS = {
        "the", "of", "to", "with", "and", "or", "for", "this"
    };

    /**
     * An Intermedia wildcard that matches any sequence of
     * characters
     */
    public static final String WILD_CARD = "%";

    /**
     * An "and" join clause
     */
    public static final String AND = "and";

    /**
     * An "or" join clause
     */
    public static final String OR = "or";


    /**
     * Determine if the string represents a common word.
     * Common words, such as "the", "of", etc., should not be
     * included in search clauses.
     *
     * @param s a search keyword
     * @return true if <code>s</code> is a common word, false otherwise
     */
    public boolean isCommonWord(String s) {
        for(int i=0; i<COMMON_WORDS.length; i++) {
            if(COMMON_WORDS[i].equalsIgnoreCase(s))
                return true;
        }

        return false;
    }


    /**
     * Convert some keywords which are typed in by the user to
     * an Intermedia search clause. Clean out all the special
     * characters, and remove common words. Surround each word
     * with the '%' wildcard. For example, the search string
     * <code>"cat, fish and bird"</code> will be converted to
     * <code>"%cat% and %fish% and %bird%"</code>
     *
     * @param words a string which contains some search keywords
     */
    public static String createIntermediaClause(String words) {
        return createIntermediaClause(words, AND, WILD_CARD);
    }

    /**
     * Convert some keywords which are typed in by the user to
     * an Intermedia search clause. Clean out all the special
     * characters, and remove common words. Surround each word
     * with the '%' wildcard. For example, the search string
     * <code>"cat, fish and bird"</code> will be converted to
     * <code>"%cat% and %fish% and %bird%"</code>
     *
     * @param words a string which contains some search keywords
     * @param joinClause a string which will be used to combine the
     *   keywords. Should be <code>"and"</code>, <code>"or"</code>,
     *   or a similar boolean operator.
     * @param wildcard a wildcard which will be appened to the right
     *   and to the left of each keyword
     */
    public static String createIntermediaClause(
                                                String words, String joinClause, String wildcard
                                                ) {

        StringTokenizer tokenizer = new StringTokenizer(words, SPECIAL_CHARS);
        StringBuffer result = new StringBuffer();
        String nextJoin = "";

        // Normalize the join clause
        StringBuffer joinBuf = new StringBuffer();
        joinBuf.append(" ").append(joinClause.trim()).append(" ");
        joinClause = joinBuf.toString();

        // Append words
        while(tokenizer.hasMoreTokens()) {
            result.append(nextJoin);
            String token = tokenizer.nextToken();
            result.append(token).append(wildcard);
            nextJoin = joinClause;
        }

        return result.toString();
    }
}
