/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.ui.lifecycle;

import com.arsdigita.bebop.BoxPanel;
import com.arsdigita.bebop.ColumnPanel;
import com.arsdigita.bebop.FormData;
import com.arsdigita.bebop.FormProcessException;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.event.FormInitListener;
import com.arsdigita.bebop.event.FormProcessListener;
import com.arsdigita.bebop.event.FormSectionEvent;
import com.arsdigita.bebop.form.Hidden;
import com.arsdigita.bebop.form.Submit;
import com.arsdigita.bebop.parameters.BigDecimalParameter;
import com.arsdigita.bebop.parameters.NotNullValidationListener;
import com.arsdigita.cms.SecurityManager;
import com.arsdigita.cms.lifecycle.PhaseDefinition;
import com.arsdigita.cms.ui.CMSForm;
import com.arsdigita.cms.ui.FormSecurityListener;
import com.arsdigita.domain.DataObjectNotFoundException;

import java.math.BigDecimal;

/**
 * This class handles the deleting of a phase definition.
 *
 * @author Jack Chung (flattop@arsdigita.com)
 * @version $Revision: #6 $ $Date: 2004/04/07 $
 */
class DeletePhaseForm extends CMSForm
        implements FormProcessListener, FormInitListener {
    public static final String versionId =
        "$Id: //cms/dev/src/com/arsdigita/cms/ui/lifecycle/DeletePhaseForm.java#6 $" +
        "$Author: dennis $" +
        "$DateTime: 2004/04/07 16:07:11 $";

    private final PhaseRequestLocal m_phase;

    private Hidden m_id;
    private Submit m_deleteWidget;
    private Submit m_cancelWidget;

    /**
     * @param m The phase selection model. This tells the form which
     * phase definition is selected.
     */
    public DeletePhaseForm(final PhaseRequestLocal phase) {
        super("PhaseDefinitionDelete");

        m_phase = phase;

        m_id = new Hidden(new BigDecimalParameter("id"));
        add(m_id);
        m_id.addValidationListener(new NotNullValidationListener());

        BoxPanel buttons = new BoxPanel(BoxPanel.HORIZONTAL);
        m_deleteWidget = new Submit("delete");
        m_deleteWidget.setButtonLabel("Delete");
        m_deleteWidget.setClassAttr("deletePhase");
        buttons.add(m_deleteWidget);

        m_cancelWidget = new Submit("cancel");
        m_cancelWidget.setButtonLabel("Cancel");
        m_cancelWidget.setClassAttr("canceldeletePhase");
        buttons.add(m_cancelWidget);

        add(buttons, ColumnPanel.CENTER|ColumnPanel.FULL_WIDTH);

        addInitListener(this);

        addSubmissionListener
            (new FormSecurityListener(SecurityManager.LIFECYCLE_ADMIN));

        addProcessListener(this);
    }

    /**
     * Returns true if this form was cancelled.
     *
     * @param s The page state
     * @return true if the form was cancelled, false otherwise
     */
    public boolean isCancelled(PageState s) {
        return m_cancelWidget.isSelected(s);
    }

    /**
     * Form process listener.
     * Deletes a phase definition
     *
     * @param e The form process event
     * @exception FormProcessException
     */
    public final void process(final FormSectionEvent e) 
            throws FormProcessException {
        final FormData data = e.getFormData();
        final BigDecimal key = (BigDecimal) data.get(m_id.getName());

        // Check if the object is already deleted for double click
        // protection.
        try {
            PhaseDefinition phaseDefinition = new PhaseDefinition(key);
            phaseDefinition.delete();
        } catch (DataObjectNotFoundException ex) {
            // Just ignore it since it is already deleted.
        }

    }

    /**
     * Init listener.
     * gets the id of the selected phase definition
     *
     * @param e The form init event
     */
    public final void init(final FormSectionEvent e) {
        final FormData data = e.getFormData();
        final PageState state = e.getPageState();

        final BigDecimal id =  m_phase.getPhase(state).getID();

        data.put(m_id.getName(), id);
    }
}
