/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.ui.category;

import com.arsdigita.bebop.ColumnPanel;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.form.Submit;
import com.arsdigita.categorization.Category;
import com.arsdigita.categorization.CategoryCollection;
import com.arsdigita.cms.SecurityManager;
import com.arsdigita.cms.ui.CategoryForm;
import com.arsdigita.cms.ui.FormSecurityListener;
import com.arsdigita.kernel.ACSObject;
import com.arsdigita.toolbox.ui.Cancellable;

import java.math.BigDecimal;

import org.apache.log4j.Logger;

/**
 * A form which edits secondary parents
 *
 * @author Michael Pih
 * @author Stanislav Freidin
 * @version $Revision: #17 $ $DateTime: 2004/04/07 16:07:11 $
 */
public class LinkForm extends CategoryForm implements Cancellable {
    private final static Logger s_log = Logger.getLogger(LinkForm.class);

    private final CategoryRequestLocal m_category;
    private final Submit m_cancelButton;

    public LinkForm(final CategoryRequestLocal category) {
        super("LinkForm");

        m_category = category;

        m_cancelButton = new Submit("Finish");
        add(m_cancelButton, ColumnPanel.FULL_WIDTH | ColumnPanel.LEFT);

        setAssignedCaption("Linked Categories");

        addSubmissionListener
            (new FormSecurityListener(SecurityManager.CATEGORY_ADMIN));
    }

    public final boolean isCancelled(final PageState state) {
        return m_cancelButton.isSelected(state);
    }

    /**
     * Load all categories which are assigned to the current item.
     */
    protected void initAssignedCategories(PageState state, CategoryMap m) {
        final Category category = m_category.getCategory(state);

        final BigDecimal parentID = category.getDefaultParentCategory().getID();
        CategoryCollection links = category.getParents();

        while ( links.next() ) {
            Category cat = links.getCategory();

            if ( !cat.getID().equals(parentID) ) {
                m.add(cat);
            }
        }
        links.close();
    }

    /**
     * Assign a secondary parent.
     */
    public void assignCategory(PageState state, Category category) {
        final Category child = m_category.getCategory(state);
        if (category.canEdit()) {
            category.addChild(child);
            category.save();
        }
    }

    /**
     * Unassign a secondary parent.
     */
    public void unassignCategory(PageState state, Category category) {
        final Category child = m_category.getCategory(state);

        if (category.canEdit()) {
            category.removeChild(child);
            category.save();
        }
    }

    /**
     * The category cannot be its own parent. Its children cannot
     * be parents either.
     */
    public Category getExcludedCategory(PageState state) {
        return m_category.getCategory(state);
    }

     /**
      *  This method returns the URL for the given item to make sure that
      *  there are not two objects in the same category with the same URL.
      */
     protected final String getItemURL(final PageState state) {
         return m_category.getCategory(state).getURL();
     }

     protected final ACSObject getObject(final PageState state) {
         return (Category) m_category.getCategory(state);
     }
}
