/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.ui.authoring;


import com.arsdigita.bebop.FormData;
import com.arsdigita.bebop.FormProcessException;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.event.FormSectionEvent;
import com.arsdigita.cms.ContentItem;
import com.arsdigita.cms.ContentPage;
import com.arsdigita.cms.ContentSection;
import com.arsdigita.cms.Folder;
import com.arsdigita.cms.ItemSelectionModel;
import com.arsdigita.cms.ui.ItemPropertySheet;
import com.arsdigita.cms.ui.workflow.WorkflowLockedComponentAccess;
import com.arsdigita.cms.util.GlobalizationUtil;
import com.arsdigita.util.Assert;
import org.apache.log4j.Logger;

import java.text.DateFormat;


/**
 * The article editing component. Consists of a display component
 * which displays the form metadata, and a form which edits the
 * metadata as well as text.
 * 
 * The {@link com.arsdigita.bebop.PropertySheet} class is often used
 * as the display component in the default authoring kit steps of
 * this class.
 *
 * @author Stanislav Freidin
 * @version $Revision: #13 $ $DateTime: 2004/04/07 16:07:11 $
 */
public class PageEdit extends SimpleEditStep {

    public static final String versionId = "$Id: //cms/dev/src/com/arsdigita/cms/ui/authoring/PageEdit.java#13 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private static Logger s_log =
        Logger.getLogger(PageEdit.class);

    //XD: The output escaping of the values of the label value pairs
    //i.e., values of the attributes name and title
    public static final boolean VALUE_OUTPUT_ESCAPE = true;

    /**
     * Construct a new PageEdit component
     *
     * @param itemModel The {@link ItemSelectionModel} which will
     *   be responsible for loading the current item
     *
     * @param parent The parent wizard which contains the form. The form
     *   may use the wizard's methods, such as stepForward and stepBack,
     *   in its process listener.
     */
    public PageEdit(ItemSelectionModel itemModel, AuthoringKitWizard parent) {
        super(itemModel, parent);

        PageEditForm form = new PageEditForm(itemModel);
        add("edit", "Edit", new WorkflowLockedComponentAccess(form, itemModel),
            form.getSaveCancelSection().getCancelButton());

        //DomainObjectPropertySheet sheet = new DomainObjectPropertySheet(itemModel);
        ItemPropertySheet sheet =
            new ItemPropertySheet(itemModel, VALUE_OUTPUT_ESCAPE);
        sheet.add((String) GlobalizationUtil.globalize("cms.ui.authoring.name").localize(),  ContentPage.NAME);
        sheet.add((String) GlobalizationUtil.globalize("cms.ui.authoring.title").localize(),  ContentPage.TITLE);
        if (!ContentSection.getConfig().getHideLaunchDate()) {
            sheet.add("Launch Date:",
                      ContentPage.LAUNCH_DATE,
                      new ItemPropertySheet.AttributeFormatter() {
                          public String format(ContentItem item,
                                               String attribute,
                                               PageState state) {
                              ContentPage page = (ContentPage) item;
                              if(page.getLaunchDate() != null) {
                                  return DateFormat.getDateInstance(DateFormat.LONG)
                                      .format(page.getLaunchDate());
                              } else {
                                  return "<i>unknown</i>";
                              }
                          }
                      });
        }
        setDisplayComponent(sheet);
    }



    /**
     * A form for editing Article items. This is your basic page editing
     * form with a file upload/text entry widget
     */
    private class PageEditForm extends BasicPageForm {

        /**
         * Construct a new PageEditForm
         *
         * @param itemModel The {@link ItemSelectionModel} which will
         *   be responsible for loading the current item
         *
         */
        public PageEditForm(ItemSelectionModel itemModel) {
            super("PageEditForm", itemModel);
        }

        // Init: load the item and preset the widgets
        public void init(FormSectionEvent e) throws FormProcessException {
            super.initBasicWidgets(e);
        }

        // Process: save fields to the database
        public void process(FormSectionEvent e) throws FormProcessException {
            PageState state = e.getPageState();
            ContentPage item = (ContentPage)super.processBasicWidgets(e);
            if ( item != null ) {
                item.save();
            }
        }

        public void validate(FormSectionEvent event) throws FormProcessException {
            PageState state = event.getPageState();
            FormData data = event.getFormData();

            ContentItem item =
                (ContentItem) getItemSelectionModel().getSelectedObject(state);
            Assert.assertNotNull(item);

            String newName = (String) data.get(BasicPageForm.NAME);
            String oldName = item.getName();

            // Validation passes if the item name is the same.
            if ( !newName.equalsIgnoreCase(oldName) ) {
                validateNameUniqueness((Folder) item.getParent(), event);
            }
        }
    }

}
