/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.ui;

import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.PaginationModelBuilder;
import com.arsdigita.bebop.Paginator;
import com.arsdigita.bebop.RequestLocal;
import com.arsdigita.bebop.SingleSelectionModel;
import com.arsdigita.cms.ContentItem;
import com.arsdigita.cms.ImageAssetCollection;
import com.arsdigita.cms.ReusableImageAsset;
import com.arsdigita.util.Assert;
import com.arsdigita.util.LockableImpl;

/**
 * Builds an {@link ImageBrowserModel} by selecting all images which match the
 * given keyword. The keyword is supplied by the <code>getSelectedKey</code>
 * method of a <code>SingleSelectionModel</code>. If the keyword is empty
 * or null, the builder will return an {@link EmptyImageBrowserModel}
 *
 * @author Stanislav Freidin (sfreidin@arsdigita.com)
 * @version $Id: //cms/dev/src/com/arsdigita/cms/ui/DefaultImageBrowserModelBuilder.java#9 $
 */
public class DefaultImageBrowserModelBuilder extends LockableImpl
    implements ImageBrowserModelBuilder, PaginationModelBuilder {

    public static final String versionId = "$Id: //cms/dev/src/com/arsdigita/cms/ui/DefaultImageBrowserModelBuilder.java#9 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private SingleSelectionModel m_keywordModel;
    private static ImageBrowserModel EMPTY_MODEL =
        new EmptyImageBrowserModel();

    private ImageBrowser m_imageBrowser;
    private RequestLocal m_size;
    private RequestLocal m_imageColl;

    private String m_context;

    /**
     * Construct a new  DefaultImageBrowserModelBuilder
     *
     * @param keywordModel The SingleSelectionModel whose getSelectedKey(state)
     *    method returns a string keyword
     * @param context the context for the retrieved items. Should be
     *   {@link ContentItem#DRAFT} or {@link ContentItem#LIVE}
     */
    public DefaultImageBrowserModelBuilder(
                                           SingleSelectionModel keywordModel, String context
                                           ) {
        super();
        m_keywordModel = keywordModel;
        m_context = context;
	m_size = new RequestLocal();
	m_imageColl = new RequestLocal();
    }

    /**
     * Construct a new  DefaultImageBrowserModelBuilder
     *
     * @param keywordModel The SingleSelectionModel whose getSelectedKey(state)
     *    method returns a string keyword
     */
    public DefaultImageBrowserModelBuilder(SingleSelectionModel keywordModel) {
        this(keywordModel, ContentItem.DRAFT);
    }

    /**
     * Construct an ImageBrowserModel for the current request
     */
    public ImageBrowserModel makeModel(ImageBrowser browser, PageState s) {
        //String key = (String)m_keywordModel.getSelectedKey(s);

	// pass through key even if null -- null key will return all rows in m_context.
        //ImageAssetCollection c = ReusableImageAsset.getReusableImagesByKeyword(key, m_context);
        return new DefaultImageBrowserModel((ImageAssetCollection) m_imageColl.get(s));
    }

    /**
     * @return the keyword selection model
     */
    public SingleSelectionModel getKeywordModel() {
        return m_keywordModel;
    }

    /**
     * @param context the new context for the items. Should be
     *   {@link ContentItem#DRAFT} or {@link ContentItem#LIVE}
     */
    public void setContext(String context) {
        Assert.assertNotLocked(this);
        m_context = context;
    }

    public int getTotalSize(Paginator paginator, PageState state) {
	
	Integer size = (Integer) m_size.get(state);
	
	if (size == null) {
	    
	    String key = (String)m_keywordModel.getSelectedKey(state);
	    ImageAssetCollection c = ReusableImageAsset.getReusableImagesByKeyword(key, m_context);
	    if (c == null) {
		return 0;
	    }

	    size = new Integer( (int) ReusableImageAsset.getReusableImagesByKeyword(key, m_context).size());
	    
	    c.setRange(new Integer(paginator.getFirst(state)),
		       new Integer(paginator.getLast(state) + 1));
	    
	    m_size.set(state, size);
	    m_imageColl.set(state, c);
	}
	
	return size.intValue();
    }
    
    public void setImageBrowser(ImageBrowser ib) {
	m_imageBrowser = ib;
    }
    /**
     * Indicates whether the paginator should be visible,
     * based on the visibility of the image browser itself.
     *
     * @return true if image browser is visible, or if the
     *         associated image browser is unknown.
     */
    public boolean isVisible(PageState state) {
	return (m_imageBrowser != null)?m_imageBrowser.isVisible(state):true;
    }
    
}
