/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.search;

import com.arsdigita.cms.ContentType;
import com.arsdigita.search.FilterType;
import com.arsdigita.search.FilterSpecification;
import com.arsdigita.search.lucene.BaseQueryEngine;
import com.arsdigita.search.lucene.UnionFilter;
import com.arsdigita.search.lucene.UnionFilter;
import com.arsdigita.search.lucene.ObjectTypeFilter;
import com.arsdigita.search.lucene.PartyFilter;
import com.arsdigita.search.lucene.TypeSpecificFilter;
import com.arsdigita.search.lucene.Document;
import com.arsdigita.search.filters.DateRangeFilterSpecification;
import com.arsdigita.search.filters.PartyFilterSpecification;
import com.arsdigita.cms.search.ContentTypeFilterSpecification;
import com.arsdigita.cms.search.ContentTypeFilterType;
import com.arsdigita.kernel.PartyCollection;

import java.util.List;
import java.util.ArrayList;
import java.util.Date;

import org.apache.lucene.search.Filter;
import org.apache.lucene.search.DateFilter;


public class LuceneQueryEngine extends BaseQueryEngine {

    protected void addFilter(List list,
                             FilterSpecification filter) {
        super.addFilter(list, filter);

        FilterType type = filter.getType();
        
        if (ContentTypeFilterType.KEY.equals(type.getKey())) {
           addContentTypeFilter(list, (ContentTypeFilterSpecification)filter);
        } else if (VersionFilterType.KEY.equals(type.getKey())) {
           addVersionFilter(list, (VersionFilterSpecification)filter);
        } else if (LastModifiedDateFilterType.KEY.equals(type.getKey())) {
            addDateRangeFilter(list, 
                               (DateRangeFilterSpecification)filter, 
                               Document.LAST_MODIFIED_DATE);
        } else if (CreationDateFilterType.KEY.equals(type.getKey())) {
            addDateRangeFilter(list, 
                               (DateRangeFilterSpecification)filter, 
                               Document.CREATION_DATE);
        } else if (LastModifiedUserFilterType.KEY.equals(type.getKey())) {
            addPartyFilter(list, (PartyFilterSpecification)filter,
                           Document.LAST_MODIFIED_PARTY);
        } else if (CreationUserFilterType.KEY.equals(type.getKey())) {
            addPartyFilter(list, (PartyFilterSpecification)filter,
                           Document.CREATION_PARTY);
        }
    }

    protected void addDateRangeFilter(List list,
                                      DateRangeFilterSpecification filter,
                                      String paramName) {
        Date startDate = filter.getStartDate();
        Date endDate = filter.getEndDate();
        if (startDate != null && endDate != null) {
            list.add(new DateFilter(paramName, startDate, endDate));
        } else if (startDate != null) {
            list.add(DateFilter.After(paramName, startDate));
        } else if (endDate != null) {
            list.add(DateFilter.Before(paramName, startDate));
        } 
    }

    protected void addPartyFilter(List list,
                                  PartyFilterSpecification filter,
                                  String filterType) {
        PartyCollection parties = filter.getParties();
        if (parties == null) {
            return;
        }
        List filters = new ArrayList();
        int count = 0;
        while (parties.next()) {
            filters.add(new PartyFilter(parties.getID(), filterType));
            count++;
        }
        list.add(new UnionFilter((Filter[])filters.toArray(new Filter[count])));
    }


    protected void addVersionFilter(List list,
                                    VersionFilterSpecification filter) {
        list.add(new TypeSpecificFilter(filter.getVersion()));
    }

    protected void addContentTypeFilter(List list,
                                        ContentTypeFilterSpecification filter) {
        List l = new ArrayList();
        ContentType[] types = filter.getTypes();
        if (types == null || types.length == 0) {
            return;
        }

        Filter[] filters = new Filter[types.length];
        for (int i = 0 ; i < types.length ; i++) {                        
            filters[i] = new ObjectTypeFilter(types[i].getAssociatedObjectType());
        }
        list.add(new UnionFilter(filters));
    }    
}
