/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.search;
import com.arsdigita.search.FilterSpecification;
import com.arsdigita.cms.ContentType;

import com.arsdigita.util.Assert;

/**
 * A filter spec for supplying a list of content types
 * to the content type filter type. 
 */
public class ContentTypeFilterSpecification extends FilterSpecification {

    public final static String TYPES = "types";
    
    /**
     * Creates a new filter restricting results to a single
     * content type.
     * @param typeName the content type name
     */
    public ContentTypeFilterSpecification(String typeName) {
        this(lookupTypes(new String[] {typeName}));
    }

    /**
     * Creates a new filter restricting results to a set
     * content types.
     * @param typeName the content type names
     */
    public ContentTypeFilterSpecification(String[] typeNames) {
        this(lookupTypes(typeNames));
    }

    /**
     * Creates a new filter restricting results to a single
     * content type.
     * @param type the content type
     */
    public ContentTypeFilterSpecification(ContentType type) {
        super(new Object[] { TYPES, new ContentType[] { type } },
              new ContentTypeFilterType());
    }

    /**
     * Creates a new filter restricting results to a set
     * content type.
     * @param type the content types
     */
    public ContentTypeFilterSpecification(ContentType[] types) {
        super(new Object[] { TYPES, types },
              new ContentTypeFilterType());
    }
    
    /**
     * Returns the set of object types to filter on
     * @return the object type
     */
    public ContentType[] getTypes() {
        return (ContentType[])get(TYPES);
    }

    private static ContentType[] lookupTypes(String[] typeNames) {
        Assert.exists(typeNames, String[].class);
        ContentType[] types = new ContentType[typeNames.length];
        for (int i = 0 ; i < typeNames.length ; i++) {
            types[i] = ContentType.findByAssociatedObjectType(typeNames[i]);
        }
        return types;
    }
}
