/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.publishToFile;

import com.arsdigita.domain.DomainCollection;
import com.arsdigita.persistence.DataCollection;
import com.arsdigita.persistence.SessionManager;

import java.math.BigDecimal;

/**
 * A collection of queue entries.
 *
 * @author <a href="mailto:dlutter@redhat.com">David Lutterkort</a>
 * @version $Id: //cms/dev/src/com/arsdigita/cms/publishToFile/QueueEntryCollection.java#6 $
 */
class QueueEntryCollection extends DomainCollection {
  
    /**
     * Produce a colection of all queue entries with the given
     * <code>itemID</code>.
     *
     * @param itemID item for which all queue entries should be listed.
     */
    public QueueEntryCollection(BigDecimal itemID) {
        super(getEntries(itemID));
    }

    /**
     * Delete the current entry in the collection. If the entry is marked as
     * <tt>in process</tt>, the entry is nothing is done.
     */
    public void delete() {
        if ( ! isInProcess() ) {
            m_dataCollection.getDataObject().delete();
        }
    }

    public BigDecimal getID() {
        return (BigDecimal)get(QueueEntry.ID);
    }
    public String getTask() {
        return (String) get(QueueEntry.TASK);
    }

    public boolean isPublishTask() {
        return QueueEntry.TASK_PUBLISH.equals(getTask());
    }
  
    public boolean isUnpublishTask() {
        return QueueEntry.TASK_UNPUBLISH.equals(getTask());
    }
  
    public boolean isRepublishTask() {
        return QueueEntry.TASK_REPUBLISH.equals(getTask());
    }
  
    public boolean isMoveTask() {
        return QueueEntry.TASK_MOVE.equals(getTask());
    }
  
    public boolean isInProcess() {
        return QueueEntry.IN_PROCESS_YES.equals(get(QueueEntry.IN_PROCESS));
    }
  
    private static DataCollection getEntries(BigDecimal itemID) {
        DataCollection result = SessionManager.getSession()
            .retrieve(QueueEntry.BASE_DATA_OBJECT_TYPE);
        result.addEqualsFilter(QueueEntry.ITEM_ID, itemID);
        return result;
    }

}
