/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.installer.xml;


import com.arsdigita.initializer.Configuration;
import com.arsdigita.initializer.InitializationException;
import com.arsdigita.kernel.BaseInitializer;
import com.arsdigita.persistence.SessionManager;
import com.arsdigita.persistence.TransactionContext;
import com.arsdigita.xml.XML;

import java.util.Iterator;
import java.util.List;


/** Parses XML file definition of content types and loads them to the
 * database.  The XML config looks like the example below, the
 * "parentType" and "name" attributes are optional, and only required
 * for creating User Defined ContentTypes. Label corresponds to
 * ContentType's label and can be multiple words, and "name" to
 * DynamicObject's name property, and must be a single word. The
 * objectType attribute is ignored for UDCTs, as it gets dynamically
 * generated.
 *
 * <b>UDCT Copyright</b>
 * <pre>
 * &lt;ccm:content-types&gt;
 *   &lt;ccm:content-type 
 *             name="Copyright"
 *             label="UDCT Copyright"
 *             parentType="com.arsdigita.cms.contenttypes.Address"
 *             classname="com.arsdigita.cms.contenttypes.Address"
 *             description="Copyright for storing copyright information" 
 *             objectType="com.arsdigita.cms.contentTypes.Address" &gt;
 *    
 *      &lt;ccm:authoring-kit&gt;
 *      &lt;/ccm:authoring-kit&gt;
 *   &lt;/ccm:content-type&gt;
 * &lt;/ccm:content-types&gt;
 *</pre>
 *
 * <b>Initializer</b>
 * <pre>
 *  init com.arsdigita.cms.installer.ContentTypeInitializer {
 *      contentTypes = {
 *          "/WEB-INF/content-types/Article.xml"
 *      };
 *  }
 *</pre>
 * @author Nobuko Asakai <nasakai@redhat.com>
 * @see XMLContentTypeHandler
 */
public class ContentTypeInitializer extends BaseInitializer {
    public static final String CONTENT_TYPES = "contentTypes";

    private Configuration m_conf = new Configuration();
    private static org.apache.log4j.Logger s_log =
        org.apache.log4j.Logger.getLogger(ContentTypeInitializer.class);

    public ContentTypeInitializer() {
        // Could make this a list later...
        m_conf.initParameter
            (CONTENT_TYPES,
             "Path to XML files that contains content type definition",
             List.class);
    }

    public Configuration getConfiguration() {
        return m_conf;
    }

    protected void doShutdown() {
        // Do nothing
    }

    protected void doStartup() throws InitializationException {
        List contentTypes = (List)m_conf.getParameter(CONTENT_TYPES);
        
        if ( contentTypes != null) {            
            Iterator i = contentTypes.iterator();
            while (i.hasNext()) {
                TransactionContext txn = SessionManager.getSession()
                    .getTransactionContext();
                txn.beginTxn();
                String xmlFile = (String)i.next();
                s_log.debug("Processing contentTypes in: " + xmlFile);
                XML.parseResource(xmlFile, new XMLContentTypeHandler());
                txn.commitTxn();
            }
        }
    }
}
