/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.dispatcher;


import java.math.BigDecimal;
import java.util.HashMap;


/**
 * <p>This class contains methods for registering and resolving {@link
 * ResourceHandler CMS resources} in a specific content section.</p>
 *
 * <p>The <tt>PageResolver</tt> includes methods for caching resource
 * mappings.</p>
 *
 * @author Michael Pih (pihman@arsdigita.com)
 * @version $Revision: #7 $ $Date: 2004/04/07 $
 */
public abstract class PageResolver {

    public static final String versionId = "$Id: //cms/dev/src/com/arsdigita/cms/dispatcher/PageResolver.java#7 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private BigDecimal m_sectionID;

    // Used for caching pages
    private HashMap m_pages;


    public PageResolver() {
        m_pages = new HashMap();
    }

    public void setContentSectionID(BigDecimal id) {
        m_sectionID = id;
    }

    protected BigDecimal getContentSectionID() {
        return m_sectionID;
    }


    /**
     * Fetch the page associated with the request URL.
     *
     * @param url The content section-relative URL stub
     * @return The resource
     */
    public ResourceHandler getPage(String url) {
        return (ResourceHandler) m_pages.get(url);
    }

    /**
     * Register a page to the content section.
     *
     * @param page The master page
     * @param url The desired URL of the page
     */
    public abstract void registerPage(ResourceHandler page, String url);


    /**
     * Register a page to the content section.
     *
     * @param page The master page
     * @param url The desired URL of the page
     */
    public abstract void unregisterPage(ResourceHandler page, String url);


    /**
     * Loads a page into the page resolver cache.
     *
     * @param url  The URL of the resource to load into the cache
     * @param page The resource
     */
    public void loadPage(String url, ResourceHandler page) {
        m_pages.put(url, page);
    }

    /**
     * Flushes a page from the page resolver cache.
     *
     * @param url The URL of the resource to remove from the cache
     */
    public void releasePage(String url) {
        m_pages.remove(url);
    }

}
