/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.dispatcher;

import com.arsdigita.cms.ContentItem;
import com.arsdigita.cms.ContentSection;
import com.arsdigita.cms.SecurityManager;
import com.arsdigita.dispatcher.Dispatcher;
import com.arsdigita.dispatcher.DispatcherChain;
import com.arsdigita.dispatcher.RequestContext;
import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.kernel.SiteNode;
import com.arsdigita.kernel.User;
import com.arsdigita.sitenode.SiteNodeRequestContext;
import com.arsdigita.util.Assert;
import java.io.IOException;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * <b><font color="red">Unsupported</font></b> Refactored content
 * section dispatcher (under development).
 *
 * @author Karl Goldstein (karlg@arsdigita.com)
 * @version $Revision: #8 $ $DateTime: 2004/04/07 16:07:11 $
 **/
public class ContentSectionDispatcher implements Dispatcher {

    public static final String versionId = "$Id: //cms/dev/src/com/arsdigita/cms/dispatcher/ContentSectionDispatcher.java#8 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    static final String CONTENT_ITEM = "com.arsdigita.cms.dispatcher.item";

    static final String CONTENT_SECTION = "com.arsdigita.cms.dispatcher.section";

    private DispatcherChain dispatcherChain = new DispatcherChain();

    public ContentSectionDispatcher() {

        dispatcherChain.addChainedDispatcher(new CMSDispatcher(true));
        dispatcherChain.addChainedDispatcher(new FileDispatcher());
        dispatcherChain.addChainedDispatcher(new ItemDispatcher());
        dispatcherChain.addChainedDispatcher(new CMSDispatcher());
    }

    public void dispatch(HttpServletRequest request,
                         HttpServletResponse response,
                         RequestContext context)
        throws IOException, ServletException {

        setContentSection(request, (SiteNodeRequestContext) context);
        dispatcherChain.dispatch(request, response, context);
    }

    /**
     * Fetches the content section from the request attributes.
     *
     * @param request The HTTP request
     * @return The content section
     * @pre ( request != null )
     */
    public static ContentSection getContentSection(HttpServletRequest request) {
        return (ContentSection) request.getAttribute(CONTENT_SECTION);
    }

    /**
     * Fetches the content item from the request attributes.
     *
     * @param request The HTTP request
     * @return The content item
     * @pre ( request != null )
     */
    public static ContentItem getContentItem(HttpServletRequest request) {
        return (ContentItem) request.getAttribute(CONTENT_ITEM);
    }

    /**
     * Looks up the current content section using the remaining URL
     * stored in the request context object and the SiteNode class.
     *
     * @param url The section URL stub
     * @return The current Content Section
     */
    private void setContentSection(HttpServletRequest request,
                                   SiteNodeRequestContext actx)
        throws ServletException {

        try {

            // Fetch the current site node from the request context;
            SiteNode sn = actx.getSiteNode();
            ContentSection section = ContentSection.getSectionFromNode(sn);
            Assert.assertNotNull(section);

            request.setAttribute(CONTENT_SECTION, section);

        } catch (DataObjectNotFoundException e) {

            throw new ServletException(e);
        }
    }

    /**
     * Checks that the current user has permission to access the admin pages.
     **/
    public static boolean checkAdminAccess(HttpServletRequest request,
                                           ContentSection section) {

        User user = Utilities.getCurrentUser(request);

        SecurityManager sm = new SecurityManager(section);

        return sm.canAccess(user, SecurityManager.ADMIN_PAGES);
    }
}
