/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.contenttypes.ui;

import com.arsdigita.bebop.Component;
import com.arsdigita.bebop.GridPanel;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.event.PrintEvent;
import com.arsdigita.bebop.event.PrintListener;
import com.arsdigita.cms.ItemSelectionModel;
import com.arsdigita.cms.contenttypes.ContentGroup;
import com.arsdigita.cms.contenttypes.ContentGroupContainer;
import com.arsdigita.cms.ui.authoring.AuthoringKitWizard;
import com.arsdigita.cms.ui.authoring.BasicItemForm;
import com.arsdigita.cms.ui.authoring.SimpleEditStep;
import com.arsdigita.cms.util.GlobalizationUtil;


/**
 * Authoring step to edit the simple attributes of the Brand content
 * type (and its subclasses). The attributes edited are 
 * 'smallImage', 'largeImage', 'shortDescription', and 'longDescription'
 * <p>
 * This authoring step replaces the 
 * <code>com.arsdigita.ui.authoring.PageEdit</code> step for this type.
 */
public abstract class ContentGroupPropertiesStep extends SimpleEditStep {

    private static final org.apache.log4j.Logger s_log =
        org.apache.log4j.Logger.getLogger(ContentGroupPropertiesStep.class);

    /** The name of the editing sheet added to this step */
    public static String EDIT_SHEET_NAME = "edit";

    public ContentGroupPropertiesStep(ItemSelectionModel itemModel,
                                      AuthoringKitWizard parent) {
        super(itemModel, parent);

        BasicItemForm editSheet;

        editSheet = getPropertyForm(itemModel);
        add( EDIT_SHEET_NAME, "Edit", editSheet,
             editSheet.getSaveCancelSection().getCancelButton() );
        
        setDisplayComponent( getPropertySheet( itemModel ) );
    }


    /**
     * Returns a component that displays the properties of the
     * brand specified by the ItemSelectionModel passed in.
     * @param itemModel The ItemSelectionModel to use
     * @pre itemModel != null
     * @return A component to display the state of the basic properties
     *  of the release
     */
    public Component getPropertySheet(ItemSelectionModel itemModel) {
        GridPanel panel = new GridPanel(2);
        panel.add(new Label(GlobalizationUtil.globalize("cms.contenttypes.ui.content_group_name")));
        panel.add(new Label(new PrintListener() {
                public void prepare(PrintEvent e) {
                    Label label = (Label)e.getTarget();
                    ContentGroupContainer item = 
                        (ContentGroupContainer) getItemSelectionModel()
                        .getSelectedObject(e.getPageState());
                    ContentGroup group = 
                        item.getContentGroup(getMainAttributeName());
                    if (group == null) {
                        label.setLabel("None", e.getPageState());
                        label.setFontWeight(Label.ITALIC);
                    } else {
                        label.setLabel(group.getName(), e.getPageState());
                    }
                }
            }));

        panel.add(new ContentGroupItemList(getItemSelectionModel(), 
                                           getMainAttributeName()), 
                  GridPanel.FULL_WIDTH);
                  
        return panel;
    }


    /**
     *  This is the name of the attribute that should be used when processin
     *  the form.  This is needed because the classes are created through
     *  reflection so we cannot just pass this in through the constructor
     */
    public abstract String getMainAttributeName();

    /**
     * Allow subclasses to override the generated form.
     * Used to override the 
     * ContentGroupPropertyForm.getSearchContentType() method
     * to restrict the search widget to one ContentType.
     **/
    protected ContentGroupPropertyForm getPropertyForm(ItemSelectionModel itemModel) {
        return new ContentGroupPropertyForm( itemModel, 
                                             getMainAttributeName() );
    }
}
