/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms;

import com.arsdigita.util.Assert;
import org.apache.log4j.Logger;


/**
 * <p>The entry point into all the global state that CCM CMS code expects to
 * have available to it when running, e.g. the current content section,
 * current item
 *
 * <p>This is a session object that provides an environment in which
 * code can execute. The CMSContext contains all session-specific
 * variables.  One session object is maintained per thread.</p>
 *
 * <p>Accessors of this class will assert that the item it returned is
 * not null. If the caller wants to handle the case where an item is
 * null explicitly, then use the hasContentItem and hasContentSection
 * methods first.
 *
 * @author Daniel Berrange
 * @see com.arsdigita.kernel.KernelContext
 * @see com.arsdigita.cms.CMS
 */
public final class CMSContext {
    public static final String versionId =
        "$Id: //cms/dev/src/com/arsdigita/cms/CMSContext.java#9 $" +
        "$Author: dennis $" +
        "$DateTime: 2004/04/07 16:07:11 $";

    private static final Logger s_log = Logger.getLogger(CMSContext.class);

    private ContentSection m_section = null;
    private ContentItem m_item = null;
    private SecurityManager m_security = null;

    CMSContext() {
        // Empty
    }

    public final String getDebugInfo() {
        final String info = "Current state of " + this + ":\n" +
            "           getContentSection() -> " + getContentSection() + "\n" +
            "              getContentItem() -> " + getContentItem() + "\n" +
            "          getSecurityManager() -> " + getSecurityManager();

        return info;
    }

    final CMSContext copy() {
        final CMSContext result = new CMSContext();

        result.m_section = m_section;
        result.m_item = m_item;
        result.m_security = m_security;

        return result;
    }

    /**
     * Checks if a content section is available
     * @return true if a content section is available
     */
    public final boolean hasContentSection() {
        return m_section != null;
    }

    /**
     * Gets the current content section
     * @pre hasContentSection() == true
     * @return the currently selected content section
     */
    public final ContentSection getContentSection() {
        Assert.assertNotNull(m_section, "section");
        return m_section;
    }

    /**
     * Sets the current content section
     * @param section the new content section
     */
    final void setContentSection(final ContentSection section) {
        m_section = section;

        if (s_log.isDebugEnabled()) {
            s_log.debug("Content section set to " + section);
        }
    }

    /**
     * Checks if a content item is available
     * @return true if a content item is available
     */
    public final boolean hasContentItem() {
        return m_item != null;
    }

    /**
     * Returns the current content item
     * @pre hasContentItem() == true
     * @return the current content item
     */
    public final ContentItem getContentItem() {
        Assert.assertNotNull(m_item, "item");
        return m_item;
    }

    /**
     * Sets the current content item
     * @param item the new content item
     */
    final void setContentItem(final ContentItem item) {
        m_item = item;

        if (s_log.isDebugEnabled()) {
            s_log.debug("Content item set to " + item);
        }
    }

    /**
     * Checks if there is a CMS <code>SecurityManager</code> for this
     * session.
     *
     * @see com.arsdigita.cms.SecurityManager
     * @return <code>true</code> if a security manager is available
     */
    public final boolean hasSecurityManager() {
        return m_security != null;
    }

    /**
     * Returns the current security manager.
     *
     * @return the current security manager
     */
    public final SecurityManager getSecurityManager() {
        Assert.exists(m_security, SecurityManager.class);

        return m_security;
    }

    final void setSecurityManager(final SecurityManager security) {
        m_security = security;

        if (s_log.isDebugEnabled()) {
            s_log.debug("Security manager set to " + security);
        }
    }
}
