/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.ui.authoringkit;


import com.arsdigita.bebop.ColumnPanel;
import com.arsdigita.bebop.Form;
import com.arsdigita.bebop.FormData;
import com.arsdigita.bebop.FormProcessException;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.SaveCancelSection;
import com.arsdigita.bebop.SingleSelectionModel;
import com.arsdigita.bebop.event.FormInitListener;
import com.arsdigita.bebop.event.FormProcessListener;
import com.arsdigita.bebop.event.FormSectionEvent;
import com.arsdigita.bebop.event.FormSubmissionListener;
import com.arsdigita.bebop.form.Hidden;
import com.arsdigita.bebop.form.Submit;
import com.arsdigita.bebop.form.TextField;
import com.arsdigita.bebop.parameters.BigDecimalParameter;
import com.arsdigita.bebop.parameters.NotNullValidationListener;
import com.arsdigita.bebop.parameters.StringParameter;
import com.arsdigita.cms.AuthoringKit;
import com.arsdigita.cms.ContentType;
import com.arsdigita.cms.dispatcher.Utilities;
import com.arsdigita.cms.ui.type.ContentTypeRequestLocal;
import com.arsdigita.cms.util.GlobalizationUtil;
import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.util.UncheckedWrapperException;

import java.math.BigDecimal;


/**
 * This class contains a form component to edit an authoring kit
 *
 * @author Jack Chung (flattop@arsdigita.com)
 * @version $Revision: #11 $ $Date: 2004/04/07 $
 */
public class EditKit extends Form
    implements FormProcessListener, FormInitListener {

    public static final String versionId = "$Id: //cms/dev/src/com/arsdigita/cms/ui/authoringkit/EditKit.java#11 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    protected final ContentTypeRequestLocal m_type;

    protected Hidden m_id;
    protected TextField m_createComponent;
    protected SaveCancelSection  m_saveCancelSection;

    /**
     * @param types The content type selection model. This is to tell the form
     *   which content type is selected.
     */
    public EditKit(ContentTypeRequestLocal type) {
        super("AuthoringKit");

        m_type = type;

        m_id = new Hidden(new BigDecimalParameter("id"));
        add(m_id);
        m_id.addValidationListener(new NotNullValidationListener());

        add(new Label(GlobalizationUtil.globalize("cms.ui.authoringkit.createcomponent")));
        m_createComponent = new TextField(new StringParameter("createComponent"));
        m_createComponent.setSize(40);
        m_createComponent.setMaxLength(200);
        add(m_createComponent);

        m_saveCancelSection = new SaveCancelSection();
        add(m_saveCancelSection, ColumnPanel.FULL_WIDTH|ColumnPanel.CENTER);

        addProcessListener(this);
        addInitListener(this);
        addSubmissionListener(new FormSubmissionListener() {
                public void submitted(FormSectionEvent event)
                    throws FormProcessException {
                    PageState state = event.getPageState();
                    if ( isCancelled(state) ) {
                        throw new FormProcessException( (String) GlobalizationUtil.globalize("cms.ui.cancel_hit").localize());
                    }
                }
            });
    }

    /**
     * Return true if the form is cancelled, false otherwise.
     *
     * @param state The page state
     * @return true if the form is cancelled, false otherwise.
     * @pre ( state != null )
     */
    public boolean isCancelled(PageState state) {
        return m_saveCancelSection.getCancelButton().isSelected(state);
    }

    /**
     * Form process listener which updates a authoring kit
     */
    public void process(FormSectionEvent e) throws FormProcessException {
        PageState state = e.getPageState();

        FormData data = e.getFormData();
        BigDecimal key = (BigDecimal) data.get(m_id.getName());
        String createComponent = (String) data.get(m_createComponent.getName());

        AuthoringKit kit;
        try {
            kit = new AuthoringKit(key);
        } catch (DataObjectNotFoundException ex) {
            throw new UncheckedWrapperException( (String) GlobalizationUtil.globalize("cms.ui.authoringkit.content_type_id").localize() + key.toString() +
                                       " not found", ex);
        }

        kit.setCreateComponent(createComponent);
        kit.save();

        Utilities.refreshItemUI(state);
    }

    /**
     * Form init listener which initializes form values.
     */
    public void init(FormSectionEvent e) {
        FormData data = e.getFormData();
        PageState state = e.getPageState();

        AuthoringKit kit = getKit(state);
        BigDecimal id = kit.getID();
        String createComponent = kit.getCreateComponent();

        data.put(m_id.getName(), id);
        data.put(m_createComponent.getName(), createComponent);
    }

    protected AuthoringKit getKit(PageState state) {
        ContentType type = m_type.getContentType(state);
        try {
            return type.getAuthoringKit();
        } catch (DataObjectNotFoundException ex) {
            throw new UncheckedWrapperException("Authoring kit not found", ex);
        }
    }

}
