/*
 * Copyright (C) 2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.ui.authoring;

import com.arsdigita.cms.TextAsset;

import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.PropertySheet;
import com.arsdigita.bebop.PropertySheetModel;
import com.arsdigita.bebop.PropertySheetModelBuilder;
import com.arsdigita.util.LockableImpl;
import com.arsdigita.toolbox.ui.DomainObjectPropertySheet;

import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import java.util.StringTokenizer;

import com.arsdigita.domain.DomainService;
import com.arsdigita.domain.DomainObject;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.kernel.ui.DomainObjectSelectionModel;

import com.arsdigita.globalization.GlobalizedMessage;
import com.arsdigita.toolbox.util.GlobalizationUtil;
import com.arsdigita.mimetypes.MimeType;

import com.arsdigita.bebop.table.TableCellRenderer;
import com.arsdigita.bebop.table.TableModel;
import com.arsdigita.bebop.table.TableModelBuilder;
import com.arsdigita.bebop.Table;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.Component;

/**
 *  This extends DomainObjectPropertySheet and has a lot of duplicate
 *  code from it because it uses so many private inner classes
 */
public class TextAssetBodyPropertySheet extends DomainObjectPropertySheet {

    public static final String versionId = "$Id: //cms/dev/src/com/arsdigita/cms/ui/authoring/TextAssetBodyPropertySheet.java#6 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    /**
     * Construct a new TextAssetBodyPropertySheet
     *
     * @param objModel The selection model which feeds domain objects to this
     *    property sheet.
     *
     */
    public TextAssetBodyPropertySheet(DomainObjectSelectionModel objModel) {
        super(objModel);
        setModelBuilder(new TMBAdapter(new DomainObjectModelBuilder()));
        getColumn(1).setCellRenderer(new TextAssetBodyLabelCellRenderer());
    }

    // Build up the object properties model from the iterator over all properties
    private static class TextAssetBodyPropertiesModel implements PropertySheetModel {

        public final static String MIME_TYPE_KEY = 
            TextAssetBodyLabelCellRenderer.MIME_TYPE_KEY;

        private TextAsset m_obj;
        private PageState m_state;
        private Iterator m_props;
        private Property m_current;
        private static String ERROR =
            "No current property. Make sure that nextRow() was called at least once.";

        public TextAssetBodyPropertiesModel(TextAsset obj, Iterator props, 
                                            PageState state) {
            m_obj = obj;
            m_props = props;
            m_state = state;
            m_current = null;
        }

        public boolean nextRow() {
            if(!m_props.hasNext()) {
                return false;
            }

            m_current = (Property)m_props.next();
            return true;
        }

        /**
         *  @deprecated use getGlobalizedLabel() instead
         */
        public String getLabel() {
            return getGlobalizedLabel().getKey();
        }

        public GlobalizedMessage getGlobalizedLabel() {
            if(m_current == null) {
                throw new IllegalStateException(ERROR);
            }
            return m_current.getGlobalizedLabel();
        }

        public String getValue() {
            return getObjectValue().toString();
        }

        public Object getObjectValue() {
            if(m_current == null) {
                throw new IllegalStateException(ERROR);
            }

            if ((TextAsset.CONTENT.equals(m_current.getAttribute()) ||
                MIME_TYPE_KEY.equals(m_current.getAttribute())) &&
                m_obj != null && m_obj.getText() != null 
                && m_obj.getMimeType() != null) {
                return m_obj;
            } else {
                return m_current.getFormatter()
                    .format(m_obj, m_current.getAttribute(), m_state);
            }
        }

        public String getAttribute() {
            return m_current.getAttribute();
        }
    }

    // Builds an TextAssetBodyPropertiesModel
    private static class DomainObjectModelBuilder extends LockableImpl
        implements PropertySheetModelBuilder {

        public PropertySheetModel makeModel(PropertySheet sheet, PageState state) {
            TextAssetBodyPropertySheet s = (TextAssetBodyPropertySheet)sheet;
            return new TextAssetBodyPropertiesModel 
                ((TextAsset)s.getObjectSelectionModel().getSelectedObject(state),
                 s.properties(), state );
        }
    }


    // These are both from PropertySheet

    // Convert a PropertySheetModelBuilder to a TableModelBuilder
    private static class TMBAdapter
        extends LockableImpl implements TableModelBuilder {

        private PropertySheetModelBuilder m_builder;

        public TMBAdapter(PropertySheetModelBuilder b) {
            m_builder = b;
        }

        public TableModel makeModel(Table t, PageState s) {
            return new TableModelAdapter
                ((TextAssetBodyPropertiesModel)m_builder.makeModel((PropertySheet)t, s));
        }

        public void lock() {
            m_builder.lock();
            super.lock();
        }
    }

    // Wraps a PropertySheetModel
    private static class TableModelAdapter implements TableModel {

        private TextAssetBodyPropertiesModel m_model;
        private int m_row;

        public TableModelAdapter(TextAssetBodyPropertiesModel model) {
            m_model = model;
            m_row = -1;
        }

        public int getColumnCount() { return 2; }

        public boolean nextRow() {
            m_row++;
            return m_model.nextRow();
        }

        public Object getElementAt(int columnIndex) {
            if(columnIndex == 0) {
                return m_model.getGlobalizedLabel();
            } else {
                return m_model.getObjectValue();
            }
        }

        public Object getKeyAt(int columnIndex) {
            return m_model.getAttribute();
        }

        public PropertySheetModel getPSModel() {
            return m_model;
        }
    }
}
