/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.ui.authoring;


import com.arsdigita.bebop.ColumnPanel;
import com.arsdigita.bebop.FormData;
import com.arsdigita.bebop.FormProcessException;
import com.arsdigita.bebop.FormSection;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.event.FormSectionEvent;
import com.arsdigita.bebop.parameters.DateParameter;
import com.arsdigita.bebop.parameters.ParameterModel;
import com.arsdigita.cms.ContentPage;
import com.arsdigita.cms.ContentSection;
import com.arsdigita.cms.ItemSelectionModel;
import com.arsdigita.cms.util.GlobalizationUtil;
import com.arsdigita.util.Assert;

import javax.servlet.ServletException;
import java.util.Date;

/**
 * A form for editing subclasses of ContentPage. This is just a convenience
 * class.
 *
 * @author Stanislav Freidin (stas@arsdigita.com)
 * @version $Revision: #21 $ $DateTime: 2004/04/07 16:07:11 $
 **/
public abstract class BasicPageForm extends BasicItemForm {

    private FormSection m_widgetSection;

    public static final String LAUNCH_DATE = ContentPage.LAUNCH_DATE;

    /**
     * Construct a new BasicPageForm
     *
     * @param formName the name of this form
     * @param itemModel The {@link ItemSelectionModel} which will
     *   be responsible for loading the current item
     */
    public BasicPageForm(String formName, ItemSelectionModel itemModel) {
        super(formName, itemModel);
    }

    /**
     * Construct a new BasicPageForm with nothing on it
     *
     * @param formName the name of this form
     * @param columnPanel the columnpanel of the form
     * @param itemModel The {@link ItemSelectionModel} which will
     *   be responsible for loading the current item
     */
    public BasicPageForm(String formName, 
                         ColumnPanel columnPanel, 
                         ItemSelectionModel itemModel) {
        super(formName, columnPanel, itemModel);
    }


    /**
     * Add various widgets to the form. Child classes should override
     * this method to perform all their widget-adding needs
     */
    protected void addWidgets() {
        super.addWidgets();
        if (!ContentSection.getConfig().getHideLaunchDate()) {
            add(new Label(GlobalizationUtil.
                          globalize("cms.ui.authoring.page_launch_date")));
            ParameterModel launchDateParam = new DateParameter(LAUNCH_DATE);
            com.arsdigita.bebop.form.Date launchDate
                = new com.arsdigita.bebop.form.Date(launchDateParam);
            add(launchDate);
        }
    }

    /**
     * Utility method to initialize the name/title widgets. Child classes
     * may call this method from the init listener
     *
     * @param e the {@link FormSectionEvent} which was passed to the
     *    init listener
     * @return the ContentPage instance which was extracted from the
     *   ItemSelectionModel
     */
    public ContentPage initBasicWidgets(FormSectionEvent e) {
        Assert.assertNotNull(getItemSelectionModel());

        FormData data = e.getFormData();
        PageState state = e.getPageState();
        ContentPage item =
            (ContentPage)getItemSelectionModel().getSelectedObject(state);

        if (item != null) {
            // Preset fields
            data.put(NAME, item.getName());
            data.put(TITLE, item.getTitle());
            if (!ContentSection.getConfig().getHideLaunchDate()) {
                data.put(LAUNCH_DATE, item.getLaunchDate());
            }
        }

        return item;
    }

    /**
     * Utility method to process the name/title widgets. Child classes
     * may call this method from the process listener.
     *
     * @param e the {@link FormSectionEvent} which was passed to the
     *    process listener
     */
    public ContentPage processBasicWidgets(FormSectionEvent e) {
        Assert.assertNotNull(getItemSelectionModel());

        FormData data = e.getFormData();
        PageState state = e.getPageState();
        ContentPage item =
            (ContentPage)getItemSelectionModel().getSelectedObject(state);

        if (item != null) {
            // Update attributes
            item.setName((String)data.get(NAME));
            item.setTitle((String)data.get(TITLE));
            if (!ContentSection.getConfig().getHideLaunchDate()) {
                item.setLaunchDate((Date)data.get(LAUNCH_DATE));
            }
        }

        return item;
    }

    /**
     * A utility method that will create a new item and tell the selection
     * model to select the new item. Creation components may call this method
     * in the process listener of their form. See {@link PageCreate} for
     * an example.
     *
     * @param state the current page state
     * @return the new content item (or a proper subclass thereof)
     * @pre state != null
     * @post return != null
     */
    public ContentPage createContentPage(PageState state) 
          throws FormProcessException {

        ItemSelectionModel m = getItemSelectionModel();
        Assert.assertNotNull(m);

        ContentPage item = null;

        // Create new item
        try {
            item = (ContentPage) m.createItem();
        } catch (ServletException ex) {
            throw new 
                FormProcessException((String)GlobalizationUtil.globalize("cms.ui.authoring.couldnt_create_contentpage").localize(),  ex);
        }

        // Make sure the item will be remembered across requests
        m.setSelectedObject(state, item);

        return item;
    }

}
