/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.ui;

import com.arsdigita.bebop.Component;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.Link;
import com.arsdigita.bebop.Page;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.SimpleContainer;
import com.arsdigita.bebop.TabbedPane;
import com.arsdigita.bebop.event.ActionEvent;
import com.arsdigita.bebop.event.ActionListener;
import com.arsdigita.bebop.event.PrintEvent;
import com.arsdigita.bebop.event.PrintListener;
import com.arsdigita.bebop.parameters.BigDecimalParameter;
import com.arsdigita.cms.ContentItem;
import com.arsdigita.cms.ContentSection;
import com.arsdigita.cms.ContentType;
import com.arsdigita.cms.Folder;
import com.arsdigita.cms.dispatcher.CMSPage;
import com.arsdigita.cms.util.GlobalizationUtil;
import com.arsdigita.kernel.KernelHelper;
import com.arsdigita.kernel.User;
import com.arsdigita.kernel.ui.ACSObjectSelectionModel;
import com.arsdigita.util.Assert;
import com.arsdigita.ui.DebugPanel;
import com.arsdigita.web.Web;

import java.math.BigDecimal;

/**
 * <p>The Content Center page.</p>
 *
 * @author Jack Chung (flattop@arsdigita.com)
 * @author Michael Pih (pihman@arsdigita.com)
 * @version $Revision: #21 $ $DateTime: 2004/04/07 16:07:11 $
 */
public class WorkspacePage extends CMSPage implements ActionListener {
    public static final String versionId =
        "$Id: //cms/dev/src/com/arsdigita/cms/ui/WorkspacePage.java#21 $" +
        "$Author: dennis $" +
        "$DateTime: 2004/04/07 16:07:11 $";

    private final static String XSL_CLASS = "CMS Admin";

    private TabbedPane m_tabbedPane;

    private TasksPanel m_tasks;
    private ItemSearch m_search;
    private ACSObjectSelectionModel m_typeSel;
    private ACSObjectSelectionModel m_sectionSel;

    public static final String CONTENT_TYPE = "type_id";
    public static final String CONTENT_SECTION = "section_id";

    /**
     * Construct a new WorkspacePage
     */
    public WorkspacePage() {
        super(new Label( GlobalizationUtil.globalize
                         ("cms.ui.content_center")),
              new SimpleContainer());

        setClassAttr("cms-admin");

        BigDecimalParameter typeId = new BigDecimalParameter(CONTENT_TYPE);
        addGlobalStateParam(typeId);
        m_typeSel = new ACSObjectSelectionModel
            (ContentType.class.getName(), ContentType.BASE_DATA_OBJECT_TYPE, typeId);

        BigDecimalParameter sectionId = new BigDecimalParameter
            (CONTENT_SECTION);
        addGlobalStateParam(sectionId);
        m_sectionSel = new ACSObjectSelectionModel
            (ContentSection.class.getName(), ContentSection.BASE_DATA_OBJECT_TYPE, sectionId);

        add(new WorkspaceContextBar());
        add(new GlobalNavigation());

        m_tasks = getTasksPane(m_typeSel, m_sectionSel);
        m_search = getSearchPane();

        m_tabbedPane = createTabbedPane();
        m_tabbedPane.setIdAttr("page-body");
        add(m_tabbedPane);

        add(new DebugPanel());
    }

    /**
     * Creates, and then caches, the Tasks pane. Overriding this
     * method to return null will prevent this tab from appearing.
     **/
    protected TasksPanel getTasksPane(ACSObjectSelectionModel typeModel, ACSObjectSelectionModel sectionModel) {
        if (m_tasks == null) {
            m_tasks = new TasksPanel(typeModel,sectionModel);
        }
        return m_tasks;
    }

    /**
     * Creates, and then caches, the Search pane. Overriding this
     * method to return null will prevent this tab from appearing.
     **/
    protected ItemSearch getSearchPane() {
        if (m_search == null) {
            m_search = new ItemSearch(ContentItem.DRAFT);
        }

        return m_search;
    }


    private SimpleContainer makeHeader() {
        PrintListener l = new PrintListener() {
                public void prepare(PrintEvent event) {
                    PageState state = event.getPageState();
                    Link link = (Link) event.getTarget();

                    User user = KernelHelper.getCurrentUser(state.getRequest());

                    link.setChild(new Label(user.getName()));
                    link.setTarget("/pvt/");
                }
            };

        SimpleContainer sc = new SimpleContainer();
        Label welcomeLabel = new Label(GlobalizationUtil.globalize("cms.ui.welcome"));
        Link nameLink = new Link(l);

        sc.add(welcomeLabel);
        sc.add(nameLink);
        return sc;

    }


    /**
     * Created the TabbedPane to use for this page. Sets the class
     * attribute for this tabbed pane. The default implementation uses a
     * {@link com.arsdigita.bebop.TabbedPane} and sets the class
     * attribute to "CMS Admin." This implementation also adds tasks,
     * content sections, and search panes.
     *
     *<p>
     *
     * Developers can override this method to add only the tabs they
     * want, or to add additional tabs after the default CMS tabs are
     * added.
     **/
    protected TabbedPane createTabbedPane() {
        TabbedPane pane = new TabbedPane();
        pane.setClassAttr(XSL_CLASS);
        addToPane(pane, "Tasks/Sections", getTasksPane(m_typeSel, m_sectionSel));
        addToPane(pane, "Search", getSearchPane());
        pane.addActionListener(this);
        return pane;
    }


    /**
     * Adds the specified component, with the specified tab name, to the
     * tabbed pane only if it is not null.
     *
     * @param pane The pane to which to add the tab
     * @param tabName The name of the tab if it's added
     * @param comp The component to add to the pane
     **/
    protected void addToPane(TabbedPane pane, String tabName, Component comp) {
        if (comp != null) {
            pane.addTab(tabName, comp);
        }
    }


    /**
     * When a new tab is selected, reset the state of the
     * formerly-selected pane.
     *
     * @param event The event fired by selecting a tab
     */
    public void actionPerformed(ActionEvent event) {
        PageState state = event.getPageState();
        Component pane = m_tabbedPane.getCurrentPane(state);

        if ( pane == m_tasks ) {
            m_tasks.reset(state);
        } else if ( pane == m_search ) {
            m_search.reset(state);
        }
    }
}
