/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.lifecycle;

import com.arsdigita.cms.ContentBundle;
import com.arsdigita.cms.ContentItem;
import com.arsdigita.cms.ContentPage;
import com.arsdigita.cms.ContentSection;
import com.arsdigita.cms.ui.ContentItemPage;
import com.arsdigita.dispatcher.DispatcherHelper;
import com.arsdigita.domain.DomainObjectFactory;
import com.arsdigita.persistence.OID;
import com.arsdigita.util.Assert;
import com.arsdigita.web.URL;
import org.apache.log4j.Logger;

/**
 * This class makes an item live while it is in a lifecycle.
 *
 * @author Michael Pih
 * @author Stanislav Freidin
 * @version $Id: //cms/dev/src/com/arsdigita/cms/lifecycle/PublishLifecycleListener.java#16 $
 */
public class PublishLifecycleListener implements LifecycleListener {
    public static final String versionId =
        "$Id: //cms/dev/src/com/arsdigita/cms/lifecycle/PublishLifecycleListener.java#16 $" +
        "$Author: dennis $" +
        "$DateTime: 2004/04/07 16:07:11 $";

    private static final Logger s_log = Logger.getLogger
        (PublishLifecycleListener.class);

    /**
     * Invoked when a lifecycle or phase ends. The content item will
     * no longer be live.
     *
     * @param event The lifecycle event
     */
    public void begin(LifecycleEvent event) {
        s_log.info("PublishLifecycleListener begin is called. \n" +
                   event.toString() + "\n" +
                   event.getOID().toString());

        final ContentItem item = getContentItem(event.getOID());

        if (Assert.isEnabled()) {
            Assert.falsity(item instanceof ContentBundle,
                           "Content bundles cannot be published");
        }

        // Make the pending version live. Hack: this code should work
        // regardless of which item the lifecycle is attached to.

        ContentItem master = null;
        ContentItem pending = null;

        if (item.isDraftVersion()) {
            master = item;
            pending = master.getPublicVersion();

            if (Assert.isEnabled()) {
                Assert.equal(ContentItem.PENDING, pending.getVersion());
            }
        } else {
            // Item is already live or is pending.
            // ContentItem.promotePendingVersion does the right thing for both
            pending = item;
            master = pending.getDraftVersion();

            Assert.unequal(master, pending);
            Assert.equal(ContentItem.DRAFT, master.getVersion());
        }

        master.promotePendingVersion(pending);
    }

    /**
     * Invoked when a lifecycle or phase ends.
     *
     * @param event The lifecycle event
     */
    public void end(LifecycleEvent event) {
        s_log.info("PublishLifecycleListener end is called. \n" +
                    event.toString() + "\n" +
                    event.getOID().toString());

        ContentItem item = getContentItem(event.getOID());
        ContentItem master = null;

        if (item.isLiveVersion()) {
            // Get the master item for this version
            master = item.getWorkingVersion();
        } else {
            master = item;
        }

        if (master.isLive()) {
            master.setLive(null);
        }

    }

    private ContentItem getContentItem(OID oid) {
        ContentItem item = (ContentItem)DomainObjectFactory.newInstance(oid);
        return item;
    }

}
