/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.installer;

import com.arsdigita.cms.dispatcher.ContentCenterDispatcher;
import com.arsdigita.cms.dispatcher.ItemDispatcher;
import com.arsdigita.initializer.Configuration;
import com.arsdigita.initializer.InitializationException;
import com.arsdigita.persistence.SessionManager;
import com.arsdigita.persistence.TransactionContext;

import org.apache.log4j.Logger;


import java.util.Collections;
import java.util.List;


/**
 * <p>The main initializer for the Content Management System.</p>
 *
 * <p>Initializes the Content Management System, including the Content Center
 * and CMS Service applications and CMS Mime Types service. This class also
 * optionally initializes user-defined content types and user-defined content
 * sections.</p>
 *
 * @author Michael Pih (pihman@arsdigita.com)
 * @version $Revision: #46 $ $Date: 2004/04/07 $
 * @since ACS 5.0
 */
public class Initializer extends com.arsdigita.kernel.BaseInitializer {

    private static Logger s_log = Logger.getLogger(Initializer.class);

    private final static String CACHE_ITEMS        
        = "cacheItems";
    private static final String UPDATE_MASTER
        = "updateMasterObject";
    private final static String CONTENT_CENTER_MAP
        = "contentCenterMap";
    private final static String WORKSPACE            
        = "workspace";
    // Init script parameters
    private Configuration m_conf = new Configuration();


    public Initializer() throws InitializationException {
        m_conf.initParameter
            (WORKSPACE, "The name of the workspace package instance",
             String.class);
        m_conf.initParameter
            (CACHE_ITEMS, 
             "Enable caching of content items", 
             Boolean.class,
             Boolean.TRUE);
        m_conf.initParameter
            (UPDATE_MASTER,
             "If true, attempts to recursively set the correct master object for " +
             "all content items within the system.",
             Boolean.class,
             Boolean.FALSE);
        m_conf.initParameter
            (CONTENT_CENTER_MAP,
             "XML Mapping of the content center tabs to " +
             "URLs, see ContentCenterDispatcher",
             String.class,
             ContentCenterDispatcher.DEFAULT_MAP_FILE);
    }

    public Configuration getConfiguration() {
        return m_conf;
    }


    /**
     * Check if CMS package type exists. If not, then:
     *
     * <ol>
     *   <li>create CMS package type</li>
     *   <li>create Workspace package type and instance</li>
     *   <li>create CMS Service package type and instance</li>
     * </ol>
     */
    protected void doStartup() {

        TransactionContext txn =
            SessionManager.getSession().getTransactionContext();
        txn.beginTxn();

        try {

            final String workspaceURL = (String) m_conf
                .getParameter(WORKSPACE);
            final String contentCenterMap = (String)m_conf
                .getParameter(CONTENT_CENTER_MAP);

            // Update master object if upgrading from old versioning
            // XXX: shouldn't we just gut this section (and
            // VersioningUpgrader)? It is an upgrade fix from 5.1 or
            // earlier, and relying on VersionedACSObject is
            // deprecated 
            final boolean updateMaster = 
                ((Boolean)m_conf.getParameter(UPDATE_MASTER)).booleanValue();
            if (updateMaster) {
                VersioningUpgrader.updateMasterObject();
            }

            // XXX: ItemDispatcher is no longer used. Is the following
            // still a valid enterprise.init parameter? Do we need t
            // oset ContentSectionServlet.s_cacheItems instead of the
            // below (which is currently always true), or does this go
            // away entirely?
            final boolean cacheItems = 
                ((Boolean)m_conf.getParameter(CACHE_ITEMS)).booleanValue();
            s_log.debug("Set cache items to " + cacheItems);
            ItemDispatcher.setCacheItems(cacheItems);

            ContentCenterSetup centerSetup = new ContentCenterSetup(
                workspaceURL,
                contentCenterMap);

            centerSetup.run();

        } finally {
            txn.commitTxn();
        }
    }

    protected void doShutdown() {}

    
    
}
