/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms;

import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.SingleSelectionModel;
import com.arsdigita.bebop.parameters.BigDecimalParameter;
import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.kernel.ACSObject;
import com.arsdigita.kernel.ui.ACSObjectSelectionModel;
import com.arsdigita.util.UncheckedWrapperException;
import org.apache.log4j.Logger;

import javax.servlet.ServletException;
import java.math.BigDecimal;

/**
 * <p>Loads a subclass of a {@link com.arsdigita.cms.ContentItem} from
 * the database.  This model should be used as a parameter to the
 * constructor of authoring kit components.</p>
 *
 * <p>It is possible to instantiate this model with a {@link
 * com.arsdigita.cms.ContentType} as a constructor parameter. In this
 * case, the model will only instantiate items that have are of the
 * specified content type, or one of it subclasses.</p>
 *
 * @author Stanislav Freidin (stas@arsdigita.com)
 * @version $Revision: #16 $ $DateTime: 2004/04/07 16:07:11 $
 * @see com.arsdigita.kernel.ui.ACSObjectSelectionModel
 * @see com.arsdigita.bebop.SingleSelectionModel
 */
public class ItemSelectionModel extends ACSObjectSelectionModel {

    public static final String versionId = "$Id: //cms/dev/src/com/arsdigita/cms/ItemSelectionModel.java#16 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private BigDecimal m_typeId;

    private static final Logger s_log =
        Logger.getLogger(ItemSelectionModel.class);

    /**
     * Construct a new <code>ItemSelectionModel</code>.
     * This model will produce instances of <code>ContentItem</code>
     * by automatically instantiating the correct Java subclass using
     * the {@link com.arsdigita.domain.DomainObjectFactory}.
     *
     * @param parameter The state parameter which should be used to store
     *   the object ID
     */
    public ItemSelectionModel(BigDecimalParameter parameter) {
        this(null, null, parameter);
    }

    /**
     * Construct a new <code>ItemSelectionModel</code>.
     * This model will produce instances of <code>ContentItem</code>
     * by automatically instantiating the correct Java subclass using
     * the {@link com.arsdigita.domain.DomainObjectFactory}.
     *
     * @param parameterName The name of the state parameter which will
     *    be used to store the object ID.
     */
    public ItemSelectionModel(String parameterName) {
        this(null, null, new BigDecimalParameter(parameterName));
    }

    /**
     * Construct a new <code>ItemSelectionModel</code>.
     * This model will produce instances of <code>ContentItem</code>
     * by automatically instantiating the correct Java subclass using
     * the {@link com.arsdigita.domain.DomainObjectFactory}.
     *
     * @param model The {@link SingleSelectionModel} which will supply
     *    a {@link BigDecimal} ID of the currently selected item
     */
    public ItemSelectionModel(SingleSelectionModel model) {
        this(null, null, model);
    }

    /**
     * Construct a new <code>ItemSelectionModel</code>
     *
     * @param type The content type for the items this model will generate
     *
     * @param parameterName The name of the state parameter which will
     *    be used to store the item.
     */
    public ItemSelectionModel(ContentType type, String parameterName) {
        this(type, new BigDecimalParameter(parameterName));
    }

    /**
     * Construct a new <code>ItemSelectionModel</code>
     *
     * @param type The content type for the items this model will generate
     *
     * @param parameter The state parameter which should be used by this item
     *
     */
    public ItemSelectionModel(ContentType type, BigDecimalParameter parameter) {
        super(type.getClassName(), type.getAssociatedObjectType(), parameter);
        m_typeId = type.getID();
    }

    /**
     * Construct a new <code>ItemSelectionModel</code>
     *
     * @param type The content type for the items this model will generate
     *
     * @param model The {@link SingleSelectionModel} which will supply
     *    a {@link BigDecimal} id of the currently selected object
     *
     */
    public ItemSelectionModel(ContentType type, SingleSelectionModel model) {
        super(type.getClassName(), type.getAssociatedObjectType(), model);
        m_typeId = type.getID();
    }

    /**
     * Construct a new <code>ItemSelectionModel</code>
     *
     * @param itemClass The name of the Java class which represents
     *    the content item. Must be a subclass of ContentItem. In
     *    addition, the class must have a constructor with a single
     *    OID parameter.
     * @param objectType The name of the persistence metadata object type
     *    which represents the content item. In practice, will often be
     *    the same as the itemClass.
     * @param parameterName The name of the state parameter which will
     *    be used to store the item.
     */
    public ItemSelectionModel(String itemClass, String objectType,
                              String parameterName) {
        super(itemClass, objectType, new BigDecimalParameter(parameterName));
    }

    /**
     * Construct a new <code>ItemSelectionModel</code>
     *
     * @param itemClass The name of the Java class which represents
     *    the content item. Must be a subclass of ContentItem. In
     *    addition, the class must have a constructor with a single
     *    OID parameter.
     * @param objectType The name of the persistence metadata object type
     *    which represents the content item. In practice, will often be
     *    the same as the itemClass.
     * @param parameter The state parameter which should be used by this item
     */
    public ItemSelectionModel(String itemClass, String objectType,
                              BigDecimalParameter parameter) {
        super(itemClass, objectType, parameter);
    }

    /**
     * Construct a new <code>ItemSelectionModel</code>
     *
     * @param itemClass The name of the Java class which represents
     *    the content item. Must be a subclass of ContentItem. In
     *    addition, the class must have a constructor with a single
     *    OID parameter.
     * @param objectType The name of the persistence metadata object type
     *    which represents the content item. In practice, will often be
     *    the same as the itemClass.
     * @param model The {@link SingleSelectionModel} which will supply
     *    a {@link BigDecimal} id of the currently selected object
     *
     */
    public ItemSelectionModel(String itemClass, String objectType,
                              SingleSelectionModel model) {
        super(itemClass, objectType, model);
    }

    /**
     * A utility function which creates a new item with the given ID.
     * Uses reflection to create the instance of the class supplied
     * in the constructor to this ItemSelectionModel.
     *
     * @param id The id of the new item -- this is now ignored
     * @return The newly created item
     * @deprecated use createItem() instead
     */
    public ContentItem createItem(BigDecimal id) throws ServletException {
        return (ContentItem)createACSObject();
    }


    /**
     * A utility function which creates a new item.
     * Uses reflection to create the instance of the class supplied
     * in the constructor to this ItemSelectionModel.
     *
     * @return The newly created item
     */
    public ContentItem createItem() throws ServletException {
        return (ContentItem)createACSObject();
    }


    /**
     * A convenience method that gets the currently selected object
     * and casts it to a <code>ContentItem</code>
     *
     * @param s the current page state
     * @return the currently selected <code>ContentItem</code>, or null
     *   if no item was selected.
     */
    public final ContentItem getSelectedItem(PageState s) {
        return (ContentItem)getSelectedObject(s);
    }

    /**
     * A utility function which creates a new item with the given ID.
     * Uses reflection to create the instance of the class supplied
     * in the constructor to this ItemSelectionModel.
     *
     * @param id The id of the new item -- this is now ignored
     * @return The newly created item
     * @deprecated Use createACSObject() instead
     */
    public ACSObject createACSObject(BigDecimal id) throws ServletException {
        return createACSObject();
    }

    /**
     * A utility function which creates a new item.
     * Uses reflection to create the instance of the class supplied
     * in the constructor to this ItemSelectionModel.
     *
     * @return The newly created item
     */
    public ACSObject createACSObject() throws ServletException {
        ContentType type = getContentType();
        ContentItem item = (ContentItem) super.createACSObject();

        if  ( type != null ) {
            item.setContentType(type);
        }
        return item;
    }

    /**
     * @return The content type of the items which are produced by
     *   this model, or null if the content type has not been specified
     *   in the constructor.
     */
    public ContentType getContentType() {

        ContentType type = null;

        if ( m_typeId != null ) {
            try {
                type = new ContentType(m_typeId);
            } catch (DataObjectNotFoundException e) {
                s_log.error("Error locating content type with id " +
                            m_typeId, e);
                throw new UncheckedWrapperException(e);
            }
        }

        return type;
    }

}
