/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms;

import com.arsdigita.cms.lifecycle.LifecycleDefinition;
import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.domain.DomainObject;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.OID;

import java.math.BigDecimal;

/**
 * This class associates {@link com.arsdigita.cms.ContentSection
 * content sections} and {@link com.arsdigita.cms.ContentType content
 * types} with particular publication lifecycles.
 *
 * @author Jack Chung (flattop@arsdigita.com)
 * @author Michael Pih (pihman@arsdigita.com)
 * @version $Revision: #8 $ $Date: 2004/04/07 $
 */
public class ContentTypeLifecycleDefinition extends DomainObject {

    public static final String versionId = "$Id: //cms/dev/src/com/arsdigita/cms/ContentTypeLifecycleDefinition.java#8 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";


    public static final String BASE_DATA_OBJECT_TYPE =
        "com.arsdigita.cms.ContentTypeLifecycleDefinition";

    protected static final String SECTION_ID = "sectionId";
    protected static final String CONTENT_TYPE_ID = "contentTypeId";
    protected static final String LIFECYCLE_DEFINITION_ID = "lifecycleDefinitionId";


    protected ContentTypeLifecycleDefinition() {
        super(BASE_DATA_OBJECT_TYPE);
    }

    protected ContentTypeLifecycleDefinition(OID oid)
        throws DataObjectNotFoundException {
        super(oid);
    }

    protected ContentTypeLifecycleDefinition(DataObject obj) {
        super(obj);
    }


    protected BigDecimal getContentSectionID() {
        return (BigDecimal) get(SECTION_ID);
    }

    protected void setContentSection(ContentSection section) {
        set(SECTION_ID, section.getID());
    }

    protected BigDecimal getContentTypeID() {
        return (BigDecimal) get(CONTENT_TYPE_ID);
    }

    protected void setContentType(ContentType type) {
        set(CONTENT_TYPE_ID, type.getID());
    }

    protected BigDecimal getLifecycleDefinitionID() {
        return (BigDecimal) get(LIFECYCLE_DEFINITION_ID);
    }

    protected void setLifecycleDefinition(LifecycleDefinition definition) {
        set(LIFECYCLE_DEFINITION_ID, definition.getID());
    }

    /**
     * Get the default associated lifecycle definition for a content type in a
     * particular content section
     */
    public static LifecycleDefinition getLifecycleDefinition(ContentSection section,
                                                             ContentType type) {
        try {
            OID oid = new OID(BASE_DATA_OBJECT_TYPE);
            oid.set(SECTION_ID, section.getID());
            oid.set(CONTENT_TYPE_ID, type.getID());

            ContentTypeLifecycleDefinition assn =
                new ContentTypeLifecycleDefinition(oid);
            BigDecimal lifecycleDefinitionID = assn.getLifecycleDefinitionID();

            return new LifecycleDefinition
                (new OID(LifecycleDefinition.BASE_DATA_OBJECT_TYPE,
                         lifecycleDefinitionID));

        } catch (DataObjectNotFoundException e) {
            return null;
        }

    }

    /**
     * Associated a default lifecycle definition for a content type in a
     * particular content section.  If this association already exists, the
     * previous association will be updated.
     * @return true is association is added, false if updated
     */
    public static boolean updateLifecycleDefinition(ContentSection section,
                                                    ContentType type,
                                                    LifecycleDefinition lifecycle) {

        try {
            OID oid = new OID(BASE_DATA_OBJECT_TYPE);
            oid.set(SECTION_ID, section.getID());
            oid.set(CONTENT_TYPE_ID, type.getID());

            ContentTypeLifecycleDefinition assn =
                new ContentTypeLifecycleDefinition(oid);
            assn.setLifecycleDefinition(lifecycle);
            assn.save();
            return false;

        } catch (DataObjectNotFoundException e) {
            //this association does not exist
            ContentTypeLifecycleDefinition assn =
                new ContentTypeLifecycleDefinition();
            assn.setContentSection(section);
            assn.setContentType(type);
            assn.setLifecycleDefinition(lifecycle);
            assn.save();
            return true;
        }
    }



    /**
     * Remove the default lifecycle definition association for a content type in
     * a particular content section.
     * @return true if association is deleted, false otherwise
     */
    public static boolean removeLifecycleDefinition(ContentSection section,
                                                    ContentType type) {
        try {
            OID oid = new OID(BASE_DATA_OBJECT_TYPE);
            oid.set(SECTION_ID, section.getID());
            oid.set(CONTENT_TYPE_ID, type.getID());

            ContentTypeLifecycleDefinition assn =
                new ContentTypeLifecycleDefinition(oid);
            assn.delete();
            return true;
        } catch (DataObjectNotFoundException e) {
            //don't do anything since it does not exist
            return false;
        }

    }

}
