/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms;

import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.domain.DomainObjectFactory;
import com.arsdigita.persistence.DataCollection;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.OID;
import com.arsdigita.persistence.SessionManager;
import com.arsdigita.persistence.metadata.Property;
import com.arsdigita.util.Assert;
import com.arsdigita.versioning.VersionedACSObject;

import java.math.BigDecimal;

/**
 * This class associates an Article and an Image with a particular
 * caption.
 *
 * @author Jack Chung (flattop@arsdigita.com)
 * @version $Revision: #16 $ $Date: 2004/04/07 $
 */
public class ArticleImageAssociation extends ContentItem {

    public static final String versionId = "$Id: //cms/dev/src/com/arsdigita/cms/ArticleImageAssociation.java#16 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";
    public static final String BASE_DATA_OBJECT_TYPE =
        "com.arsdigita.cms.ArticleImageAssociation";

    protected static final String ARTICLE = "captionArticle";
    protected static final String IMAGE = "imageAsset";
    protected static final String CAPTION = "caption";
    protected static final String ARTICLE_ID = "articleId";
    protected static final String IMAGE_ID = "imageId";


    /**
     * Default constructor.
     **/
    public ArticleImageAssociation() {
        super(BASE_DATA_OBJECT_TYPE);
    }

    /**
     * Constructor. The contained <code>DataObject</code> is retrieved
     * from the persistent storage mechanism with an <code>OID</code>
     * specified by <i>oid</i>.
     *
     * @param oid The <code>OID</code> for the retrieved
     * <code>DataObject</code>.
     **/
    public ArticleImageAssociation(OID oid) throws DataObjectNotFoundException {
        super(oid);
    }
    public ArticleImageAssociation(DataObject obj) {
        super(obj);
    }


    public ArticleImageAssociation(String type) {
        super(type);
    }

    /**
     *  This returns the association object that is specified by
     *  the passed in IDs or it returns null if no such association exists
     */
    public static ArticleImageAssociation retrieveAssociation
        (BigDecimal articleID, BigDecimal imageID) {
        DataCollection collection = SessionManager.getSession().retrieve
            (BASE_DATA_OBJECT_TYPE);
        collection.addEqualsFilter(ARTICLE_ID, articleID);
        collection.addEqualsFilter(IMAGE_ID, imageID);
        // no deleted associations, please
        collection.addEqualsFilter(VersionedACSObject.IS_DELETED, new BigDecimal(0));
        if (collection.next()) {
            ArticleImageAssociation association =
                new ArticleImageAssociation(collection.getDataObject());
            collection.close();
            return association;
        }
        return null;
    }

    /**
     *  This returns true if the image is associated with at least one article.
     */
    public static boolean imageHasDirectAssociation
        (BigDecimal imageID) {
        DataCollection collection = SessionManager.getSession().retrieve
            (BASE_DATA_OBJECT_TYPE);
        collection.addEqualsFilter(IMAGE_ID, imageID);
        // no deleted associations, please
        collection.addEqualsFilter(VersionedACSObject.IS_DELETED, new BigDecimal(0));
        if (collection.next()) {
            collection.close();
            return true;
        }
        return false;
    }

    /**
     *  This returns true if the image is associated with at least one article.
     */

    public static boolean imageHasAssociation
        (BigDecimal imageID) {
        try {
            ImageAsset asset = (ImageAsset) DomainObjectFactory.newInstance
                (new OID(ImageAsset.BASE_DATA_OBJECT_TYPE, imageID));
            return asset == null ? false : imageHasAssociation(asset);
        } catch (DataObjectNotFoundException e) {
            // can't find asset, return false
            return false;
        }
    }
    /**
     *  This returns true if the image is associated with at least one article, checking both liveand draft versions
     */
    public static boolean imageHasAssociation
        (ImageAsset image) {
        Assert.assertNotNull(image);
        boolean returnValue = imageHasDirectAssociation(image.getID());
        if (!returnValue) {
            if (!image.getVersion().equals(ContentItem.DRAFT)) {
                ContentItem item = image.getWorkingVersion();
                if (item != null)
                    returnValue = imageHasDirectAssociation(item.getID());
            }
        }
        if (!returnValue) {
            if (!image.getVersion().equals(ContentItem.PENDING)) {
                ItemCollection pendingVersions = image.getPendingVersions();
                while(pendingVersions.next()) {
                    ContentItem item = pendingVersions.getContentItem();
                    returnValue = returnValue || imageHasDirectAssociation(item.getID());
                }
            }
        }
        if (!returnValue) {
            if (!image.getVersion().equals(ContentItem.LIVE)) {
                ContentItem item = image.getLiveVersion();
                if (item != null)
                    returnValue = imageHasDirectAssociation(item.getID());
            }
        }

        DataCollection collection = SessionManager.getSession().retrieve
            (BASE_DATA_OBJECT_TYPE);
        collection.addEqualsFilter(IMAGE_ID, image.getID());
        // no deleted associations, please
        collection.addEqualsFilter(VersionedACSObject.IS_DELETED, new BigDecimal(0));
        if (collection.next()) {
            collection.close();
            return true;
        }
        return false;
    }

    public BigDecimal getArticleID() {
        return (BigDecimal) get(ARTICLE_ID);
    }

    public Article getArticle() {
        DataCollection col = SessionManager.getSession().retrieve(ARTICLE);
        if (col.next()) {
            Article art = new Article(col.getDataObject());
            col.close();
            return art;
        }
        return null;
    }

    public void setArticle(Article article) {
        setAssociation(ARTICLE, article);
    }

    public BigDecimal getImageID() {
        return (BigDecimal) get(IMAGE_ID);
    }

    public ImageAsset getImage() {
        return (ImageAsset) DomainObjectFactory.
            newInstance((DataObject) get(IMAGE));
    }

    public void setImage(ImageAsset image) {
        setAssociation(IMAGE, image);
    }

    public String getCaption() {
        return (String) get(CAPTION);
    }

    public void setCaption(String caption) {
        set(CAPTION, caption);
    }

    /**
     * Auto-publish the associated ReusableImageAssociation if it is not yet live
     *
     * @param source the source CustomCopy item
     * @param property the property to copy
     * @param copier a temporary class that is able to copy a child item
     *   correctly.
     * @return true if the property was copied; false to indicate
     *   that regular metadata-driven methods should be used
     *   to copy the property.
     */
    public boolean copyProperty(final CustomCopy source,
				final Property property,
				final ItemCopier copier) {
	String attribute = property.getName();
	// don't copy these attributes. they're controlled by explicit associations
	if (IMAGE_ID.equals(attribute) || ARTICLE_ID.equals(attribute)) {
	    return true;
	}

        if (copier.getCopyType() == ItemCopier.VERSION_COPY
                && IMAGE.equals(attribute)) {
	    ImageAsset image = ((ArticleImageAssociation)source).getImage();
	    if (image != null) {
		ImageAsset liveImage = (ImageAsset) image.getLiveVersion();
		if (liveImage == null) {
		    liveImage = (ImageAsset) image.createLiveVersion();
		}
	    }
	    // This method only makes sure that the ReusableImageAsset
            // is auto-published. It still returns  false so that the
            // copier can generate PublishedLink objects appropriately.
	    return false;
        }
	return super.copyProperty(source, property, copier);
    }

}
