/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.ui.lifecycle;

import com.arsdigita.bebop.ActionLink;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.SimpleContainer;
import com.arsdigita.bebop.Table;
import com.arsdigita.bebop.event.ActionEvent;
import com.arsdigita.bebop.event.ActionListener;
import com.arsdigita.cms.ContentItem;
import com.arsdigita.cms.ContentSection;
import com.arsdigita.cms.lifecycle.Lifecycle;
import com.arsdigita.cms.ui.BaseItemPane;
import com.arsdigita.cms.ui.ContentItemPage;
import com.arsdigita.cms.ui.item.ContentItemRequestLocal;
import com.arsdigita.toolbox.ui.ActionGroup;
import com.arsdigita.toolbox.ui.PropertyList;
import com.arsdigita.toolbox.ui.Section;
import com.arsdigita.web.RedirectSignal;
import com.arsdigita.web.URL;
import org.apache.log4j.Logger;

import java.text.DateFormat;

/**
 * This class contains the component which displays the information
 * for a particular lifecycle, with the ability to edit and delete.
 * This information also includes the associated phases for this
 * lifecycle, also with the ability to add, edit, and delete.
 *
 * @author Michael Pih
 * @author Jack Chung
 * @author Xixi D'Moon &lt;xdmoon@redhat.com&gt;
 * @author Justin Ross &lt;jross@redhat.com&gt;
 * @version $Id: //cms/dev/src/com/arsdigita/cms/ui/lifecycle/ItemLifecycleItemPane.java#13 $
 */
class ItemLifecycleItemPane extends BaseItemPane {
    public static final String versionId =
        "$Id: //cms/dev/src/com/arsdigita/cms/ui/lifecycle/ItemLifecycleItemPane.java#13 $" +
        "$Author: dennis $" +
        "$DateTime: 2004/04/07 16:07:11 $";

    private static final Logger s_log = Logger.getLogger
        (ItemLifecycleItemPane.class);

    private final ContentItemRequestLocal m_item;
    private final LifecycleRequestLocal m_lifecycle;
    private final SimpleContainer m_detailPane;

    public ItemLifecycleItemPane(final ContentItemRequestLocal item,
                                 final LifecycleRequestLocal cycle) {
        m_item = item;
        m_lifecycle = cycle;

        m_detailPane = new SimpleContainer();
        add(m_detailPane);
        setDefault(m_detailPane);

        m_detailPane.add(new SummarySection());
        m_detailPane.add(new PhaseSection());
    }

    private class SummarySection extends Section {
        public SummarySection() {
            setHeading(new Label(gz("cms.ui.lifecycle.details")));

            final ActionGroup group = new ActionGroup();
            setBody(group);

            group.setSubject(new Properties());
            group.addAction(new UnpublishLink());
        }

        private class Properties extends PropertyList {
            protected final java.util.List properties(final PageState state) {
                final java.util.List props = super.properties(state);
                final Lifecycle cycle = m_lifecycle.getLifecycle(state);

                final DateFormat format = DateFormat.getDateTimeInstance
                    (DateFormat.FULL, 
                     ContentSection.getConfig().getHideTimezone() ? DateFormat.SHORT : DateFormat.FULL);

                props.add(new Property(gz("cms.ui.name"),
                                       cycle.getLabel()));
                props.add(new Property
                          (gz("cms.ui.item.lifecycle.start_date"),
                           format.format(cycle.getStartDate())));

                final java.util.Date endDate = cycle.getEndDate();

                if (endDate == null) {
                    props.add(new Property
                              (gz("cms.ui.item.lifecycle.end_date"),
                               lz("cms.ui.none")));
                } else {
                    props.add(new Property
                              (gz("cms.ui.item.lifecycle.end_date"),
                               format.format(endDate)));
                }

                return props;
            }
        }
    }

    private class UnpublishLink extends ActionLink {
        UnpublishLink() {
            super(new Label(gz("cms.ui.item.lifecycle.unpublish")));

            addActionListener(new Listener());
        }

        private class Listener implements ActionListener {
            public final void actionPerformed(final ActionEvent e) {
                final PageState state = e.getPageState();
                final ContentItem item = m_item.getContentItem(state);

                item.unpublish();

                final String target = URL.getDispatcherPath() +
                    ContentItemPage.getItemURL(item,
                                               ContentItemPage.AUTHORING_TAB);

                throw new RedirectSignal(target, true);
            }
        }
    }

    private class PhaseSection extends Section {
        PhaseSection() {
            super(gz("cms.ui.lifecycle.phases"));

            final ActionGroup group = new ActionGroup();
            setBody(group);

            group.setSubject(new PhaseTable());
        }
    }

    private class PhaseTable extends Table {
        PhaseTable() {
            super(new ItemPhaseTableModelBuilder(m_lifecycle),
                  new String[] {
                      lz("cms.ui.name"),
                      lz("cms.ui.description"),
                      lz("cms.ui.item.lifecycle.start_date"),
                      lz("cms.ui.item.lifecycle.end_date")
                  });
        }
    }
}
