/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.ui.folder;

import com.arsdigita.bebop.FormProcessException;
import com.arsdigita.bebop.Page;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.event.FormSectionEvent;
import com.arsdigita.cms.Folder;
import com.arsdigita.cms.ItemSelectionModel;
import com.arsdigita.cms.ui.authoring.BasicItemForm;
import com.arsdigita.util.Assert;

/**
 * Class FolderForm implements the basic form for creating or renaming folders.
 *
 * @author Jon Orris (jorris@arsdigita.com)
 *
 * @version $Revision #1 $DateTime: 2004/04/07 16:07:11 $
 */
public abstract class FolderForm extends BasicItemForm{

    public static final String versionId = "$Id: //cms/dev/src/com/arsdigita/cms/ui/folder/FolderForm.java#8 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    FolderSelectionModel m_currentFolder;

    /**
     * Create a new folder form.
     *
     * @param name Name of the form
     * @param folder SelectionModel containing the current folder being operated on.
     *
     * @pre name != null && folder != null
     */
    public FolderForm(String name, FolderSelectionModel folder) {
        super(name, new ItemSelectionModel(Folder.BASE_DATA_OBJECT_TYPE, Folder.BASE_DATA_OBJECT_TYPE, "fldr"));
        m_currentFolder = folder;
    }

    public void register(Page p) {
        super.register(p);
        p.addComponentStateParam(this, getItemSelectionModel().getStateParameter());
    }

    /**
     * Returns true if the form submission was cancelled.
     */
    public boolean isCancelled(PageState s) {
        return getSaveCancelSection().getCancelButton().isSelected(s);
    }

    /**
     * Validates the form. Checks for name uniqueness.
     */
    public void validate(FormSectionEvent e) throws FormProcessException {
        Folder folder = (Folder) m_currentFolder.getSelectedObject(e.getPageState());
        Assert.assertNotNull(folder);
        validateNameUniqueness(folder, e);
    }

    /**
     * Updates a folder with a new parent, name, and label.
     *
     * @param folder The folder to update
     * @param parent The new parent folder. May be null.
     * @param name The new name of the folder
     * @param label The new label for the folder
     */
    final protected void updateFolder(Folder folder, Folder parent, String name, String label) {
        folder.setParent(parent);
        updateFolder(folder, name, label);
    }

    /**
     * Updates a folder with a new name and label.
     *
     * @param folder The folder to update
     * @param name The new name of the folder
     * @param label The new label for the folder
     */
    final protected void updateFolder(Folder folder, String name, String label) {
        folder.setName(name);
        folder.setLabel(label);
        folder.save();
    }

    /**
     * Returns the current folder being operated on.
     *
     * @return The current folder
     *
     * @pre state != null
     * @post return != null
     */
    final protected Folder getCurrentFolder(PageState state) {
        Folder folder = (Folder) m_currentFolder.getSelectedObject(state);
        return folder;
    }
}
