/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.ui.category;

import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.ParameterSingleSelectionModel;
import com.arsdigita.bebop.parameters.BigDecimalParameter;
import com.arsdigita.categorization.Category;
import com.arsdigita.cms.CMS;
import com.arsdigita.cms.SecurityManager;
import com.arsdigita.cms.ui.SortableList;
import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.domain.DomainObjectFactory;
import com.arsdigita.kernel.ACSObject;
import com.arsdigita.persistence.OID;
import org.apache.log4j.Logger;

import javax.servlet.ServletException;
import java.math.BigDecimal;

/**
 * This list offers the option for the code to provide the developer
 * with links to sort the given categories.
 *
 * NOTE: This UI currently does not scale well with large numbers of
 * items since it just lists all of them.  It would probably be nice
 * to integrate a paginator as well to as to allow the user to move an
 * item in large distances and to insert an item in the middle.  Right
 * now, when you add an item it is just placed at the end.  However,
 * if you want the item to appear in the middle then you must hit the
 * "up" arrow n/2 times where n is the number of items in the list.
 * This clearly is not a good setup.
 *
 * @author Randy Graebner (randyg@alum.mit.edu)
 * @version $Id: //cms/dev/src/com/arsdigita/cms/ui/category/SortableCategoryList.java#11 $
 */
abstract class SortableCategoryList extends SortableList {
    public static final String versionId =
        "$Id: //cms/dev/src/com/arsdigita/cms/ui/category/SortableCategoryList.java#11 $" +
        "$Author: dennis $" +
        "$DateTime: 2004/04/07 16:07:11 $";

    private static final Logger s_log = Logger.getLogger
        (SortableCategoryList.class);

    public final static String CHILDREN = "ch";

    private final CategoryRequestLocal m_parent;

    /**
     * This just makes a standard
     * {@link SortableList}
     */
    public SortableCategoryList(final CategoryRequestLocal parent) {
        super(new ParameterSingleSelectionModel
              (new BigDecimalParameter(CHILDREN)));

        m_parent = parent;

        setIdAttr("categorized_objects_list");
    }

    protected final Category getCategory(final PageState state) {
        return m_parent.getCategory(state);
    }

    /**
     *  This actually performs the sorting
     */
    public void respond(PageState ps) throws ServletException {
        String event = ps.getControlEventName();

        if (NEXT_EVENT.equals(event) || PREV_EVENT.equals(event)) {
            try {
                ACSObject child =
                    (ACSObject)DomainObjectFactory.newInstance
                    (new OID(ACSObject.BASE_DATA_OBJECT_TYPE,
                             new BigDecimal(ps.getControlEventValue())));
                Category parent = m_parent.getCategory(ps);

                if (CMS.getContext().getSecurityManager().canAccess
                        (SecurityManager.CATEGORY_ADMIN)) {
                    if (NEXT_EVENT.equals(event)) {
                        parent.swapWithNext(child);
                    } else {
                        parent.swapWithPrevious(child);
                    }

                    parent.save();
                }
            } catch (DataObjectNotFoundException e) {
                s_log.error("Trying to create categories with state = " + ps, e);
                throw new ServletException(e);
            }
        } else {
            super.respond(ps);
        }
    }
}
