/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.ui.category;


import com.arsdigita.bebop.Component;
import com.arsdigita.bebop.Grid;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.Link;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.RequestLocal;
import com.arsdigita.bebop.SingleSelectionModel;
import com.arsdigita.bebop.Table;
import com.arsdigita.bebop.table.TableCellRenderer;
import com.arsdigita.categorization.Category;
import com.arsdigita.cms.CMS;
import com.arsdigita.cms.ContentPage;
import com.arsdigita.cms.ContentSection;
import com.arsdigita.cms.ContentType;
import com.arsdigita.cms.dispatcher.ItemResolver;
import com.arsdigita.cms.ui.CMSContainer;
import com.arsdigita.cms.util.GlobalizationUtil;
import com.arsdigita.domain.DomainObject;
import com.arsdigita.domain.DomainObjectFactory;
import com.arsdigita.kernel.ACSObject;
import com.arsdigita.kernel.KernelHelper;
import com.arsdigita.kernel.User;
import com.arsdigita.kernel.ui.ACSObjectSelectionModel;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.DataQuery;
import com.arsdigita.persistence.OID;
import com.arsdigita.toolbox.ui.DataQueryListModelBuilder;
import com.arsdigita.util.Assert;

/**
 * Displays a list of items for the given category
 *
 * WARNING: The code to actually list the items is currently a travesty.
 * It needs to be re-written from scratch, by using custom data queries.
 */
public class CategoryItemsBrowser extends Grid {

    public static final String versionId = "$Id: //cms/dev/src/com/arsdigita/cms/ui/category/CategoryItemsBrowser.java#14 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private static final org.apache.log4j.Logger s_log =
        org.apache.log4j.Logger.getLogger(CategoryItemsBrowser.class);

    private RequestLocal m_resolver;

    private String m_context;

    /**
     * Construct a new CategoryItemsBrowser
     * <p>
     * The {@link SingleSelectionModel} which will provide the
     * current category
     *
     * @param sel the {@link ACSObjectSelectionModel} which will maintain
     *   the current category
     *
     * @param numCols the number of columns in the browser
     *
     * @param context the context for the retrieved items. Should be
     *   {@link com.arsdigita.cms.ContentItem#DRAFT} or {@link com.arsdigita.cms.ContentItem#LIVE}
     */
    public CategoryItemsBrowser(ACSObjectSelectionModel sel, int numCols,
                                String context) {
        super(null, numCols);
        super.setModelBuilder(new CategoryItemModelBuilder(sel));
        m_context = context;

        setRowSelectionModel(sel);
        setEmptyView(new Label(GlobalizationUtil.globalize
                               ("cms.ui.category.item.none")));

        // Cache the item resolver
        m_resolver = new RequestLocal() {
                public Object initialValue(PageState s) {
                    ContentSection section =
                        CMS.getContext().getContentSection();
                    final ItemResolver itemResolver = section.getItemResolver();
                    s_log.warn("Item resolver is" + itemResolver.getClass());
                    return itemResolver;
                }
            };

        setDefaultCellRenderer(new ItemSummaryCellRenderer());
    }

    /**
     * @return the current context
     */
    public String getContext() {
        return m_context;
    }

    /**
     * @param context the new context for the items. Should be
     *   {@link com.arsdigita.cms.ContentItem#DRAFT} or {@link com.arsdigita.cms.ContentItem#LIVE}
     */
    public void setContext(String context) {
        Assert.assertNotLocked(this);
        m_context = context;
    }

    /**
     * Iterates through all the children of the given Category
     */
    private class CategoryItemModelBuilder
        extends DataQueryListModelBuilder {

        private ACSObjectSelectionModel m_sel;

        public CategoryItemModelBuilder(ACSObjectSelectionModel sel) {
            super(ContentPage.QUERY_PAGE + "." + ACSObject.ID,
                  ContentPage.QUERY_PAGE);
            m_sel = sel;
        }

        public DataQuery getDataQuery(PageState s) {
            Category cat = (Category)m_sel.getSelectedObject(s);

            ContentSection section = CMS.getContext().getContentSection();
            User user = KernelHelper.getCurrentUser( s.getRequest() );
            OID oid = null;
            if (user != null) {
                oid = user.getOID();
            }
            // If the category is the root, list all items
            if(cat == null || (cat.equals(section.getRootCategory()))) {
                return ContentPage.getPagesInSectionQuery
                    (section, getContext(), oid);
            } else {
                return ContentPage.getPagesInSectionQuery
                    (section, getContext(), cat, oid);
            }
        }
    }

    /**
     * Renders a ContentItem in preview mode
     */
    private class ItemSummaryCellRenderer
        implements TableCellRenderer {

        public Component getComponent(Table table, PageState state, Object value,
                                      boolean isSelected, Object key,
                                      int row, int column) {

            if(value == null)
                return new Label("&nbsp;", false);

            DomainObject d = DomainObjectFactory.newInstance((DataObject)value);

            Assert.assertTrue(d instanceof ContentPage);
            ContentPage p = (ContentPage)d;

            CMSContainer box = new CMSContainer();
            Component c;

            ContentSection section =
                CMS.getContext().getContentSection();

            ItemResolver resolver = (ItemResolver)m_resolver.get(state);

            final String url = resolver.generateItemURL
                                     (state, p.getID(), p.getName(), section,
                                      resolver.getCurrentContext(state));
            c = new Link(p.getTitle(), url);

            c.setClassAttr("title");
            box.add(c);

            String summary = p.getSearchSummary();
            if(summary != null && summary.length() > 0) {
                c = new Label(summary);
                c.setClassAttr("summary");
                box.add(c);
            }

            ContentType t = p.getContentType();
            if(t != null) {
                c = new Label(t.getLabel());
            } else {
                c = new Label(GlobalizationUtil.globalize("cms.ui.category.item"));
            }
            c.setClassAttr("type");
            box.add(c);

            box.setClassAttr("itemSummary");

            return box;
        }
    }
}
