/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms;

import com.arsdigita.domain.DomainObject;
import com.arsdigita.kernel.ACSObject;
import com.arsdigita.persistence.OID;

import com.arsdigita.persistence.metadata.Property;

/**
 * <p>This is a supplementary class which should only be used in the
 * <code>copyProperty</code> method of {@link CustomCopy}
 * implementations. For a more detailed description, see the javadoc
 * on {@link CustomCopy#copyProperty(CustomCopy, Property,
 * ItemCopier)}.</p>
 *
 * <p>Note that the <code>copyProperty</code> method must <b>only</b> call
 * <code>copier.copy</code> in order to correctly copy some
 * content item inside the method body. An attempt to call any other method
 * to copy an item may cause disastrous results.</p>
 *
 * <p>For a correct implementation of the <code>copyProperty</code> method,
 * see {@link ContentItem#copyProperty(CustomCopy, Property, ItemCopier)}.</p>
 *
 * <p>There are not, and should not be, any public implementations of
 * this class.</p>
 *
 * @author Stanislav Freidin (sfreidin@arsdigita.com)
 * @version $Id: //cms/dev/src/com/arsdigita/cms/ItemCopier.java#9 $
 * @see CustomCopy#copyProperty(CustomCopy, Property, ItemCopier)
 */
public interface ItemCopier {

    /**
     * Denotes that this instance item copier is used for a regular
     * copy operation
     */
    public static final int PLAIN_COPY = 0;
    public static final int VERSION_COPY = 1;

    /**
     * Creates a copy, by reference or by value, of the property
     * represented in <code>object</code>.
     *
     * When called from within <code>CustomCopy.copyProperty</code>,
     * source will be the source object passed into
     * <code>copyProperty</code>, target will be <code>this</code>,
     * and <code>object</code> will be the object which is to be copied.
     *
     * @param source the <code>DomainObject</code> source (original)
     * object to which this property belongs
     * @param target the new <code>DomainObject</code> copy to which
     * the return value of this method will be attached
     * @param object the <code>DomainObject</code> property being
     * copied
     * @param prop a <code>Property</code> representing
     * <code>object</code>
     * @return <code>object</code> if <code>prop</code> is not a
     * component or a copy of <code>object</code> it is a component
     */
    DomainObject copy(final CustomCopy source,
		      final CustomCopy target,
		      final DomainObject object,
		      final Property prop);

    /**
     * Return a copy of the object identified by the specified OID.
     * The copy may not exist if the specified OID has not been copied yet;
     * in this case, the method will return <code>null</code>.
     * <p>
     * This method will typically be used in the
     * {@link ContentItem#copyProperty(ContentItem, String, ItemCopier)}
     * method.
     *
     * @param oid the OID of the object whose copy should be retrieved
     * @return the copy of the specified object, or null if the object has
     *   not been copied yet, or if the object is not convertible to a
     *  <code>DomainObject</code>.
     */
    DomainObject getCopy(OID oid);

    /**
     * Return PLAIN_COPY if the particular instance of the item
     * copier is used for making plain copies of the item. Return
     * VERSION_COPY id the copier is used to create live versions
     * for draft items.
     */
    public int getCopyType();
}
