/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms;

import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.mimetypes.MimeType;
import com.arsdigita.persistence.DataCollection;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.Filter;
import com.arsdigita.persistence.OID;
import com.arsdigita.persistence.SessionManager;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.math.BigDecimal;



/**
 * An {@link com.arsdigita.cms.Asset asset} describing a concrete
 * file, such as an image.
 *
 * @author Scott Seago (scott@arsdigita.com)
 * @version $Revision: #14 $ $DateTime: 2004/04/07 16:07:11 $
 */
public class FileAsset extends BinaryAsset {

    public static final String versionId = "$Id: //cms/dev/src/com/arsdigita/cms/FileAsset.java#14 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    public static final String BASE_DATA_OBJECT_TYPE =
        "com.arsdigita.cms.FileAsset";


    public static final String CONTENT = "content";
    public static final String HEIGHT = "height";
    public static final String WIDTH = "width";

    //public static final String MIME_JPEG = "image/jpeg";
    //public static final String MIME_GIF = "image/gif";

    /**
     * Default constructor. This creates a new text asset.
     **/
    public FileAsset() {
        super(BASE_DATA_OBJECT_TYPE);
    }

    /**
     * Constructor. The contained <code>DataObject</code> is retrieved
     * from the persistent storage mechanism with an <code>OID</code>
     * specified by <i>oid</i>.
     *
     * @param oid The <code>OID</code> for the retrieved
     * <code>DataObject</code>.
     **/
    public FileAsset(OID oid) throws DataObjectNotFoundException {
        super(oid);
    }

    /**
     * Constructor. The contained <code>DataObject</code> is retrieved
     * from the persistent storage mechanism with an <code>OID</code>
     * specified by <i>id</i> and
     * <code>FileAsset.BASE_DATA_OBJECT_TYPE</code>.
     *
     * @param id The <code>id</code> for the retrieved
     * <code>DataObject</code>.
     **/
    public FileAsset(BigDecimal id) throws DataObjectNotFoundException {
        this(new OID(BASE_DATA_OBJECT_TYPE, id));
    }

    public FileAsset(DataObject obj) {
        super(obj);
    }

    public FileAsset(String type) {
        super(type);
    }

    /**
     * @return the base PDL object type for this item. Child classes should
     *  override this method to return the correct value
     */
    public String getBaseDataObjectType() {
        return BASE_DATA_OBJECT_TYPE;
    }

    public BigDecimal getWidth() {
        return (BigDecimal) get(WIDTH);
    }

    public void setWidth(BigDecimal width) {
        set(WIDTH, width);
    }

    public BigDecimal getHeight() {
        return (BigDecimal) get(HEIGHT);
    }

    public void setHeight(BigDecimal height) {
        set(HEIGHT, height);
    }

    /**
     * Retrieves the Blob content.
     *
     * @return the Blob content
     */
    public byte[] getContent() {
        return (byte[]) get(CONTENT);
    }

    /**
     * Sets the Blob content.
     */
    protected void setContent(byte[] content) {
        set(CONTENT, content);
    }

    /**
     * Load the file asset from the specified file. Automatically guesses
     * the mime type of the file.
     *
     * @param fileName  The original name of the file
     * @param file The actual file on the server
     * @param defaultMimeType The default mime type for the file
     */
    public void loadFromFile(String fileName, File file, String defaultMimeType)
        throws IOException {

        // Guess mime type
        MimeType mime = MimeType.guessMimeTypeFromFile(fileName);

        if(mime == null && defaultMimeType != null) {
            // Set default mime type
            mime = MimeType.loadMimeType(defaultMimeType);
        }

        setMimeType(mime);

        // Extract the filename
        int i = fileName.lastIndexOf("/");
        if(i > 0) {
            fileName = fileName.substring(i+1);
        }
        i = fileName.lastIndexOf("\\");  // DOS-style
        if(i > 0) {
            fileName = fileName.substring(i+1);
        }

        setName(fileName);

        FileInputStream in = new FileInputStream(file);
        readBytes(in);
    }

    /**
     * Write the file asset content to a file.
     *
     * @param file      The file on the server to write to.
     */
    public void writeToFile(File file)
        throws IOException {
        FileOutputStream fs = new FileOutputStream(file);
        try {
            fs.write(getContent());

        } finally {
            if (null != fs) {
                fs.close();
            }
        }
    }


    /**
     * Retrieve all files in the database. Extremely expensive !
     *
     * @return a collection of FileAssets
     */
    public static FileAssetCollection getAllFiles() {
        DataCollection da = SessionManager.getSession().retrieve
            (BASE_DATA_OBJECT_TYPE);
        return new FileAssetCollection(da);
    }

    /**
     * Find all files whose name matches the specified keyword
     *
     * @param keyword a String keyword
     * @param context the context for the retrieved items. Should be
     *   {@link ContentItem#DRAFT} or {@link ContentItem#LIVE}
     * @return a collection of files whose name matches the keyword
     */
    public static FileAssetCollection getFilesByKeyword(
                                                        String keyword, String context
                                                        ) {
        FileAssetCollection c = getAllFiles();
        c.addOrder(Asset.NAME);
        Filter f;
        f = c.addFilter("name like (\'%\' || :keyword || \'%\')");
        f.set("keyword", keyword);
        f = c.addFilter("version = :version");
        f.set("version", context);
        return c;
    }

    /**
     * Find all files whose name matches the specified keyword
     *
     * @param keyword a String keyword
     * @return a collection of files whose name matches the keyword
     */
    public static FileAssetCollection getFilesByKeyword(String keyword) {
        return getFilesByKeyword(keyword, ContentItem.DRAFT);
    }

}
