/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms;

import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.domain.DomainObject;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.OID;
import com.arsdigita.workflow.simple.WorkflowTemplate;
import com.arsdigita.util.Assert;

import java.math.BigDecimal;

import org.apache.log4j.Logger;

/**
 * This class models a three-way association that represents the
 * default {@link com.arsdigita.workflow.simple.WorkflowTemplate
 * workflow template} registered to a {@link
 * com.arsdigita.cms.ContentType content type} within a {@link
 * com.arsdigita.cms.ContentSection content section}.
 *
 * @author Michael Pih (pihman@arsdigita.com)
 * @version $Id: //cms/dev/src/com/arsdigita/cms/ContentTypeWorkflowTemplate.java#10 $
 */
public class ContentTypeWorkflowTemplate extends DomainObject {
    public static final String versionId =
        "$Id: //cms/dev/src/com/arsdigita/cms/ContentTypeWorkflowTemplate.java#10 $" + 
        "$Author: dennis $" +
        "$DateTime: 2004/04/07 16:07:11 $";

    private static final Logger s_log = Logger.getLogger
        (ContentTypeWorkflowTemplate.class);

    public static final String BASE_DATA_OBJECT_TYPE =
        "com.arsdigita.cms.ContentTypeWorkflowTemplate";

    protected static final String SECTION_ID = "sectionId";
    protected static final String CONTENT_TYPE_ID = "contentTypeId";
    protected static final String WF_TEMPLATE_ID = "workflowTemplateId";


    protected ContentTypeWorkflowTemplate() {
        super(BASE_DATA_OBJECT_TYPE);
    }

    protected ContentTypeWorkflowTemplate(OID oid)
        throws DataObjectNotFoundException {
        super(oid);
    }

    protected ContentTypeWorkflowTemplate(DataObject obj) {
        super(obj);
    }

    protected BigDecimal getContentSectionID() {
        return (BigDecimal) get(SECTION_ID);
    }

    protected BigDecimal getContentTypeID() {
        return (BigDecimal) get(CONTENT_TYPE_ID);
    }

    protected BigDecimal getWorkflowTemplateID() {
        return (BigDecimal) get(WF_TEMPLATE_ID);
    }

    protected void setContentSection(ContentSection section) {
        set(SECTION_ID, section.getID());
    }

    protected void setContentType(ContentType type) {
        set(CONTENT_TYPE_ID, type.getID());
    }

    protected void setWorkflowTemplate(WorkflowTemplate template) {
        set(WF_TEMPLATE_ID, template.getID());
    }

    /**
     * Get the default associated workflow template for a content type in a
     * particular content section.
     *
     * @param section The content section
     * @param type The content type
     * @return The default workflow template, null if there is none.
     */
    public static WorkflowTemplate getWorkflowTemplate
            (final ContentSection section, final ContentType type) {
        if (Assert.isEnabled()) {
            Assert.exists(section, ContentSection.class);
            Assert.exists(type, ContentType.class);
        }

        try {
            OID oid = new OID(BASE_DATA_OBJECT_TYPE);
            oid.set(SECTION_ID, section.getID());
            oid.set(CONTENT_TYPE_ID, type.getID());

            final ContentTypeWorkflowTemplate assn =
                new ContentTypeWorkflowTemplate(oid);

            final BigDecimal id = assn.getWorkflowTemplateID();

            return new WorkflowTemplate
                (new OID(WorkflowTemplate.BASE_DATA_OBJECT_TYPE, id));
        } catch (DataObjectNotFoundException e) {
            s_log.debug("There is no default workflow template for CT " +
                    type.getLabel() +
                    " in section " +
                    section.getName());
            return null;
        }
    }


    /**
     * Associate a default workflow template for a content type in a
     * particular content section.  If this association already exists, the
     * previous association will be updated.
     *
     * @param section The content section
     * @param type The content type
     * @param template The workflow template
     * @return true is association is added, false if updated
     */
    public static boolean updateWorkflowTemplate(ContentSection section,
                                                 ContentType type,
                                                 WorkflowTemplate template) {
        try {
            OID oid = new OID(BASE_DATA_OBJECT_TYPE);
            oid.set(SECTION_ID, section.getID());
            oid.set(CONTENT_TYPE_ID, type.getID());

            ContentTypeWorkflowTemplate assn = new ContentTypeWorkflowTemplate(oid);
            assn.setWorkflowTemplate(template);
            assn.save();
            return false;

        } catch (DataObjectNotFoundException e) {
            // The association does not exist.
            ContentTypeWorkflowTemplate assn = new ContentTypeWorkflowTemplate();
            assn.setContentSection(section);
            assn.setContentType(type);
            assn.setWorkflowTemplate(template);
            assn.save();
            return true;
        }
    }

    /**
     * Remove the default workflow template association for a content type in
     * a particular content section.
     *
     * @param section The content section
     * @param type The content type
     * @return true if association is deleted, false otherwise
     */
    public static boolean removeWorkflowTemplate(ContentSection section,
                                                 ContentType type) {
        try {
            OID oid = new OID(BASE_DATA_OBJECT_TYPE);
            oid.set(SECTION_ID, section.getID());
            oid.set(CONTENT_TYPE_ID, type.getID());

            ContentTypeWorkflowTemplate assn = new ContentTypeWorkflowTemplate(oid);
            assn.delete();
            return true;
        } catch (DataObjectNotFoundException e) {
            // There is no default workflow template. Do nothing.
            return false;
        }
    }

}
