/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms;

import com.arsdigita.categorization.Category;
import com.arsdigita.domain.DomainQuery;
import com.arsdigita.persistence.DataQuery;
import com.arsdigita.persistence.Session;
import com.arsdigita.persistence.SessionManager;

import java.math.BigDecimal;

/**
 * This class represents a query which returns the ContentItems in a
 * given category. The query may return either LIVE or DRAFT
 * items. For each ContentItem returned, the fields represented in the
 * query are ItemID, Name, DisplayName, ObjectType and DefaultDomainClass.
 *
 * @see com.arsdigita.domain.DomainQuery
 **/
public class CategoryItemsQuery extends DomainQuery {
    
    private static final String ITEMS_QUERY = 
        "com.arsdigita.cms.itemsInCategory";

    public static final String CATEGORY_ID = "categoryID";
    public static final String CONTEXT = "context";
    public static final String ITEM_ID = "itemID";
    public static final String NAME = "name";
    public static final String DISPLAY_NAME = "displayName";
    public static final String OBJECT_TYPE= "objectType";
    public static final String DEFAULT_DOMAIN_CLASS= "defaultDomainClass";

    /**
     * Constructor.
     *
     * @see com.arsdigita.domain.DomainQuery
     **/
    private CategoryItemsQuery(DataQuery items) {
        super(items);
    }
    
    /**
     * Returns a CategoryItemsQuery for the given Category and default
     * LIVE context
     *
     * @param cat Category within which to retrieve items
     *
     **/
    public static CategoryItemsQuery retrieve(Category cat) {
	return retrieve(cat, ContentItem.LIVE);
    }

    /**
     * Returns a CategoryItemsQuery for the given Category and item
     * context
     *
     * @param cat Category within which to retrieve items
     * @param context Context (DRAFT or LIVE) for the items.
     *
     **/
    public static CategoryItemsQuery retrieve(Category cat, String context) {
        Session s = SessionManager.getSession();
        DataQuery dc = 
            s.retrieveQuery(ITEMS_QUERY);
        
        dc.setParameter(CATEGORY_ID, cat.getID());
	dc.setParameter(CONTEXT, (ContentItem.LIVE == context) ? 
			ContentItem.LIVE : ContentItem.DRAFT);
        
        return new CategoryItemsQuery(dc);
    }

    /**
     * Returns the value of the <i>itemID</i> property associated with
     *
     * @return the value of the itemID
     **/
    public BigDecimal getItemID() {
        return (BigDecimal)get(ITEM_ID);
    }

    /**
     * Returns the value of the <i>name</i> property associated with
     *
     * @return the value of the name
     **/
    public String getName() {
        return (String)get(NAME);
    }

    /**
     * Returns the value of the <i>displayName</i> property associated with
     *
     * @return the value of the displayName
     **/
    public String getDisplayName() {
        return (String)get(DISPLAY_NAME);
    }

    /**
     * Returns the value of the <i>objectType</i> property associated with
     *
     * @return the value of the objectType
     **/
    public String getObjectType() {
        return (String)get(OBJECT_TYPE);
    }

    /**
     * Returns the value of the <i>defaultDomainClass</i> property associated with
     *
     * @return the value of the defaultDomainClass
     **/
    public String getDefaultDomainClass() {
        return (String)get(DEFAULT_DOMAIN_CLASS);
    }
}
