/************************************************************************
 * obwrapper.cpp OpenBabel wrapper functions
 *
 * Copyright (c) 2004,2005 by Ernst-G. Schmid
 * Copyright (c) 2004,2005 by Bayer Business Services GmbH
 * for explicitly marked functions
 *
 * This file is part of the xchem::tigress project.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * lesser GNU General Public License for more details.
 ************************************************************************/
#include <list>

#if HAVE_SSTREAM
#include <sstream>
#elif HAVE_SSTREAM_H
#include <sstream.h>
#endif

#include <mol.h>
#include <fingerprint.h>
#include <obconversion.h>
#include <obiter.h>
#include "obwrapper.h"

using namespace std;
using namespace OpenBabel;
//using namespace XchemTigress;

extern "C" char *
ob_mol_to_smiles (char *molfile)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (molfile);
  string outstring;
  istringstream molstream (tmpStr);
  ostringstream smilesstream;
  char *tmpSmiles;

  conv.SetInAndOutFormats ("MDL", "SMI");
  conv.AddOption ("n", OBConversion::OUTOPTIONS);

  conv.Read (&mol, &molstream);
  conv.Write (&mol, &smilesstream);

  outstring = smilesstream.str ();

  outstring = outstring.substr (0, outstring.length () - 1);

  tmpSmiles = strdup (outstring.c_str ());

  return (tmpSmiles);
}

extern "C" char *
ob_mol_to_canonical_smiles (char *molfile, int iso_and_chiral_markings)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (molfile);
  string outstring;
  istringstream molstream (tmpStr);
  ostringstream smilesstream;
  char *tmpSmiles;

  conv.SetInAndOutFormats ("MDL", "CAN");
  conv.AddOption ("n", OBConversion::OUTOPTIONS);
  
  if(iso_and_chiral_markings != 0) {
    conv.AddOption ("i", OBConversion::OUTOPTIONS);
    }  

  conv.Read (&mol, &molstream);
  conv.Write (&mol, &smilesstream);

  outstring = smilesstream.str ();

  outstring = outstring.substr (0, outstring.length () - 1);

  tmpSmiles = strdup (outstring.c_str ());

  return (tmpSmiles);
}

extern "C" char *
ob_smiles_to_mol (char *smiles)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (smiles);
  string outstring;
  istringstream smilesstream (tmpStr);
  ostringstream molstream;
  char *tmpMolfile;

  conv.SetInAndOutFormats ("SMI", "MDL");

  conv.Read (&mol, &smilesstream);
  conv.Write (&mol, &molstream);

  outstring = molstream.str ();

  // remove the trailling $$$$\n from the SDFile
  if (outstring.find ("$$$$\n", 0) != string::npos)
    {
      outstring = outstring.substr (0, outstring.length () - 5);
    }
  else if (outstring.find ("$$$$\r\n", 0) != string::npos)
    {
      outstring = outstring.substr (0, outstring.length () - 6);
    }

  tmpMolfile = strdup (outstring.c_str ());

  return (tmpMolfile);
}

extern "C" char *
ob_mol_to_V3000 (char *molfile)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (molfile);
  string outstring;
  istringstream molstream (tmpStr);
  ostringstream V3000stream;
  char *tmpV3000;

  conv.SetInAndOutFormats ("MDL", "MDL");
  conv.AddOption ("2", OBConversion::INOPTIONS);
  conv.AddOption ("3", OBConversion::OUTOPTIONS);

  conv.Read (&mol, &molstream);
  conv.Write (&mol, &V3000stream);

  outstring = V3000stream.str ();

  // remove the trailling $$$$\n from the SDFile
  if (outstring.find ("$$$$\n", 0) != string::npos)
    {
      outstring = outstring.substr (0, outstring.length () - 5);
    }
  else if (outstring.find ("$$$$\r\n", 0) != string::npos)
    {
      outstring = outstring.substr (0, outstring.length () - 6);
    }

  tmpV3000 = strdup (outstring.c_str ());

  return (tmpV3000);
}

extern "C" char *
ob_V3000_to_mol (char *V3000)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (V3000);
  string outstring;
  istringstream V3000stream (tmpStr);
  ostringstream molstream;
  char *tmpMolfile;

  conv.SetInAndOutFormats ("MDL", "MDL");
  conv.AddOption ("3", OBConversion::INOPTIONS);
  conv.AddOption ("2", OBConversion::OUTOPTIONS);

  conv.Read (&mol, &V3000stream);
  conv.Write (&mol, &molstream);

  outstring = molstream.str ();

  // remove the trailling $$$$\n from the SDFile
  if (outstring.find ("$$$$\n", 0) != string::npos)
    {
      outstring = outstring.substr (0, outstring.length () - 5);
    }
  else if (outstring.find ("$$$$\r\n", 0) != string::npos)
    {
      outstring = outstring.substr (0, outstring.length () - 6);
    }

  tmpMolfile = strdup (outstring.c_str ());

  return (tmpMolfile);
}


/*****************************************************************
 * This function Copyright (c) 2004
 * by Bayer Business Services GmbH
 *****************************************************************/
extern "C" double
ob_molweight (char *molfile)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (molfile);
  istringstream molstream (tmpStr);
  double molweight = 0.0;

  conv.SetInAndOutFormats ("MDL", "MDL");

  conv.Read (&mol, &molstream);
  mol.AddHydrogens (false, false);

  molweight = mol.GetMolWt ();

  return (molweight);
}

/*****************************************************************
 * This function Copyright (c) 2004,2005
 * by Bayer Business Services GmbH
 *****************************************************************/
/*extern "C" char *
ob_hillformula (char *molfile)
{
    list < string > lOther;
    list < string > lChaos;
    list < string >::const_iterator e;
    string *molfmla = new string ();
    string tmpStr (molfile);
    istringstream molstream (tmpStr);
    ostringstream ossCH;
    ostringstream ossOther;
    unsigned int count;
    string currentSymbol;
    string startSymbol;
    bool hasC = false;
    OBElementTable etab;
    OBMol mol;
    OBAtom atom;
    OBConversion *conv=new OBConversion();
    char *tmpFormula;

    
    conv->SetInAndOutFormats("MDL","MDL");
    conv->Read(&mol,&molstream);
    free(conv);

    mol.AddHydrogens (false, false);

    FOR_ATOMS_OF_MOL(atom,mol)
      {
	lChaos.push_back (etab.GetSymbol (atom->GetAtomicNum ()));
      }

    lChaos.sort ();

    while (!lChaos.empty ())
      {

	startSymbol = lChaos.front ();
	count = 0;

	for (e = lChaos.begin (); e != lChaos.end (); e++)
	  {

	    currentSymbol = (*e);

	    if (currentSymbol == startSymbol)
	      {
		count++;
	      }
	    else
	      {
		break;
	      }
	  }

	if (startSymbol == "H")
	  {
	    ossCH << startSymbol;
	    if (count > 1)
	      ossCH << count;
	  }
	else if (startSymbol == "C")
	  {
	    ossCH << startSymbol;
	    if (count > 1)
	      ossCH << count;
	    hasC = true;
	  }
	else
	  {
	    ossOther.str ("");
	    ossOther << startSymbol;
	    if (count > 1)
	      ossOther << count;
	    lOther.push_back (ossOther.str ());
	  }
	
	lChaos.remove (startSymbol);
      }

    if (hasC)
      {
	molfmla->append (ossCH.str ());
      }
    else
      {
	lOther.push_back (ossCH.str ());
	lOther.sort ();
      }

    ossOther.str ("");

    for (e = lOther.begin (); e != lOther.end (); e++)
      {
	ossOther << (*e);
      }

    molfmla->append (ossOther.str ());

    tmpFormula = strdup (molfmla->c_str ());
   
    free (molfmla);
    
  return (tmpFormula);
}*/

extern "C" char *
ob_hillformula (char *molfile)
{
    
  string tmpStr (molfile);
  istringstream molstream (tmpStr);
  string molfmla;
  OBMol mol;
  OBConversion conv;
  char *tmpFormula;

  conv.SetInAndOutFormats ("MDL", "MDL");
  conv.Read (&mol, &molstream);

  mol.AddHydrogens (false, false);

  molfmla = mol.GetFormula ();

  tmpFormula = strdup (molfmla.c_str ());

  return (tmpFormula);
}

/*extern "C" char *
ob_hillformula (char *molfile)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (molfile);
  string outstring;
  istringstream molstream (tmpStr);
  ostringstream inchistream;
  char *tmpFormula;
  string::size_type first_separator_pos;
  string::size_type second_separator_pos;

  conv.SetInAndOutFormats ("MDL", "INCHI");
  conv.AddOption ("w", OBConversion::OUTOPTIONS);

  conv.Read (&mol, &molstream);

  if (mol.NumAtoms () == 0)
    {
      outstring = "";
      tmpFormula = strdup (outstring.c_str ());
      return (tmpFormula);
    }

  conv.Write (&mol, &inchistream);

  outstring = inchistream.str ();

  first_separator_pos = outstring.find ("/", 0) + 1;
  second_separator_pos = outstring.find ("/", first_separator_pos);

  outstring =
    outstring.substr (first_separator_pos,
		      (second_separator_pos - first_separator_pos));

  tmpFormula = strdup (outstring.c_str ());

  return (tmpFormula);
}*/

extern "C" unsigned int
ob_num_atoms (char *molfile)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (molfile);
  istringstream molstream (tmpStr);
  unsigned int numatoms = 0;

  conv.SetInAndOutFormats ("MDL", "MDL");

  conv.Read (&mol, &molstream);
  mol.AddHydrogens (false, false);

  numatoms = mol.NumAtoms ();

  return (numatoms);
}

extern "C" unsigned int
ob_num_heavy_atoms (char *molfile)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (molfile);
  istringstream molstream (tmpStr);
  unsigned int numheavyatoms = 0;

  conv.SetInAndOutFormats ("MDL", "MDL");

  conv.Read (&mol, &molstream);

  numheavyatoms = mol.NumHvyAtoms ();

  return (numheavyatoms);
}

extern "C" unsigned int
ob_num_bonds (char *molfile)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (molfile);
  istringstream molstream (tmpStr);
  unsigned int numbonds = 0;

  conv.SetInAndOutFormats ("MDL", "MDL");

  conv.Read (&mol, &molstream);
  mol.AddHydrogens (false, false);

  numbonds = mol.NumBonds ();

  return (numbonds);
}

extern "C" unsigned int
ob_num_rotatable_bonds (char *molfile)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (molfile);
  istringstream molstream (tmpStr);
  unsigned int numrotors = 0;

  conv.SetInAndOutFormats ("MDL", "MDL");

  conv.Read (&mol, &molstream);

  numrotors = mol.NumRotors ();

  return (numrotors);
}

extern "C" int
ob_total_charge (char *molfile)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (molfile);
  istringstream molstream (tmpStr);
  int totalcharge = 0;

  conv.SetInAndOutFormats ("MDL", "MDL");

  conv.Read (&mol, &molstream);

  totalcharge = mol.GetTotalCharge ();

  return (totalcharge);
}

extern "C" unsigned int
ob_is_nostruct (char *molfile)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (molfile);
  istringstream molstream (tmpStr);

  conv.SetInAndOutFormats ("MDL", "MDL");

  conv.Read (&mol, &molstream);

  return (mol.NumAtoms () == 0 && mol.NumBonds () == 0) ? 1 : 0;
}

extern "C" unsigned int
ob_is_chiral (char *molfile)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (molfile);
  istringstream molstream (tmpStr);
  int chiral = 0;

  conv.SetInAndOutFormats ("MDL", "MDL");

  conv.Read (&mol, &molstream);
  mol.FindChiralCenters ();

  chiral = mol.IsChiral ()? 1 : 0;

  return (chiral);
}

extern "C" int
ob_2D (char *molfile)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (molfile);
  istringstream molstream (tmpStr);
  int twod = 0;

  conv.SetInAndOutFormats ("MDL", "MDL");

  conv.Read (&mol, &molstream);

  if (mol.Has2D ())
    twod++;

  return (twod);
}

extern "C" int
ob_3D (char *molfile)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (molfile);
  istringstream molstream (tmpStr);
  int threed = 0;

  conv.SetInAndOutFormats ("MDL", "MDL");

  conv.Read (&mol, &molstream);

  if (mol.Has3D ())
    threed++;

  return (threed);
}

extern "C" char *
ob_add_hydrogens (char *molfile, int polaronly, int correct4PH)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (molfile);
  string outstring;
  istringstream molstream1 (tmpStr);
  ostringstream molstream2;
  char *tmpMolfile;

  conv.SetInAndOutFormats ("MDL", "MDL");

  conv.Read (&mol, &molstream1);

  mol.AddHydrogens (polaronly != 0, correct4PH != 0);

  conv.Write (&mol, &molstream2);

  outstring = molstream2.str ();

  // remove the trailling $$$$\n from the SDFile
  if (outstring.find ("$$$$\n", 0) != string::npos)
    {
      outstring = outstring.substr (0, outstring.length () - 5);
    }
  else if (outstring.find ("$$$$\r\n", 0) != string::npos)
    {
      outstring = outstring.substr (0, outstring.length () - 6);
    }

  tmpMolfile = strdup (outstring.c_str ());

  return (tmpMolfile);
}

extern "C" char *
ob_delete_hydrogens (char *molfile, int nonpolaronly)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (molfile);
  string outstring;
  istringstream molstream1 (tmpStr);
  ostringstream molstream2;
  char *tmpMolfile;

  conv.SetInAndOutFormats ("MDL", "MDL");

  conv.Read (&mol, &molstream1);

  if (nonpolaronly != 0)
    {
      mol.DeleteNonPolarHydrogens ();
    }
  else
    {
      mol.DeleteHydrogens ();
    }

  conv.Write (&mol, &molstream2);

  outstring = molstream2.str ();

  // remove the trailling $$$$\n from the SDFile
  if (outstring.find ("$$$$\n", 0) != string::npos)
    {
      outstring = outstring.substr (0, outstring.length () - 5);
    }
  else if (outstring.find ("$$$$\r\n", 0) != string::npos)
    {
      outstring = outstring.substr (0, outstring.length () - 6);
    }

  tmpMolfile = strdup (outstring.c_str ());

  return (tmpMolfile);
}

extern "C" char *
ob_strip_salts (char *molfile)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (molfile);
  string outstring;
  istringstream molstream1 (tmpStr);
  ostringstream molstream2;
  char *tmpMolfile;

  conv.SetInAndOutFormats ("MDL", "MDL");

  conv.Read (&mol, &molstream1);

  mol.StripSalts ();

  conv.Write (&mol, &molstream2);

  outstring = molstream2.str ();

  // remove the trailling $$$$\n from the SDFile
  if (outstring.find ("$$$$\n", 0) != string::npos)
    {
      outstring = outstring.substr (0, outstring.length () - 5);
    }
  else if (outstring.find ("$$$$\r\n", 0) != string::npos)
    {
      outstring = outstring.substr (0, outstring.length () - 6);
    }

  tmpMolfile = strdup (outstring.c_str ());

  return (tmpMolfile);
}

/*****************************************************************
 * This function Copyright (c) 2005
 * by Bayer Business Services GmbH
 *****************************************************************/
extern "C" double
ob_exactmass (char *molfile)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (molfile);
  istringstream molstream (tmpStr);
  double exactmass = 0.0;

  conv.SetInAndOutFormats ("MDL", "MDL");

  conv.Read (&mol, &molstream);
  mol.AddHydrogens (false, false);

  exactmass = mol.GetExactMass ();

  return (exactmass);
}

/*extern "C" double
ob_tanimoto_otf (char *molfile1, char* molfile2)
{
  OBMol mol1, mol2;
  OBConversion conv;
  OBFingerprint* fprint;
  string tmpStr1 (molfile1);
  string tmpStr2 (molfile2);
  string fp_mode("FP2");
  istringstream molstream1 (tmpStr1);
  istringstream molstream2 (tmpStr2);
  double tanimoto = 0.0;
  vector<unsigned int> fp1, fp2;
  fprint = OBFingerprint::FindFingerprint(fp_mode);
  
  conv.SetInAndOutFormats("MDL","MDL");
  
  conv.Read(&mol1,&molstream1);
  
  fprint->GetFingerprint(&mol1,fp1);
  
  conv.Read(&mol2,&molstream2);
  
  fprint->GetFingerprint(&mol2,fp2);  
  
  tanimoto = OBFingerprint::Tanimoto(fp1,fp2);

  return (tanimoto);
}*/

extern "C" void
ob_fp2 (char *molfile, unsigned int *fp)
{
  OBMol mol;
  OBConversion conv;
  OBFingerprint *fprint;
  string tmpStr (molfile);
  string fp_mode ("FP2");
  istringstream molstream (tmpStr);
  vector < unsigned int >vfp; //bytes
  fprint = OBFingerprint::FindFingerprint (fp_mode);
  vector<unsigned int>::iterator i;
  int j=0;

  conv.SetInAndOutFormats ("MDL", "MDL");

  conv.Read (&mol, &molstream);

  fprint->GetFingerprint (&mol, vfp); //fold to 512 bits = 64 bytes
    
    for (i = vfp.begin();i != vfp.end();i++) {
      fp[j++]=*i;
  }

}

extern "C" double
ob_tanimoto (unsigned int *fp1, unsigned int *fp2)
{
  double tanimoto = 0.0;
  const int fpsize = 64; //bytes
  vector < unsigned int >vfp1 (fpsize/4), vfp2 (fpsize/4); //unsigned int = 4 byte

  memcpy (&vfp1[0], fp1, fpsize);
  memcpy (&vfp2[0], fp2, fpsize);

  tanimoto = OBFingerprint::Tanimoto (vfp1, vfp2);

  return (tanimoto);
}

/* Copyright  The International Union of Pure and Applied Chemistry 2005: IUPAC
  International Chemical Identifier (InChI) (contact: secretariat@iupac.org) */

extern "C" char *
ob_mol_to_inchi (char *molfile)
{
  OBMol mol;
  OBConversion conv;
  string tmpStr (molfile);
  string outstring;
  istringstream molstream (tmpStr);
  ostringstream inchistream;
  char *tmpInChI;

  conv.SetInAndOutFormats ("MDL", "INCHI");
  conv.AddOption ("w", OBConversion::OUTOPTIONS);

  conv.Read (&mol, &molstream);
  conv.Write (&mol, &inchistream);

  //cout << inchistream.str();

  outstring = inchistream.str ();

  outstring = outstring.substr (0, outstring.length () - 1);

  tmpInChI = strdup (outstring.c_str ());

  return (tmpInChI);
}

extern "C" unsigned int
ob_SSS_SMARTS (const char *smarts_pattern, char *molfile)
{
  OBMol mol;
  OBConversion conv;
  OBSmartsPattern sp;
  string tmpStr (molfile);
  istringstream molstream (tmpStr);
  vector<vector<int> > maplist;

  conv.SetInAndOutFormats ("MDL", "MDL");

  conv.Read (&mol, &molstream);
  
  sp.Init(smarts_pattern);
  
  sp.Match(mol);

  maplist = sp.GetUMapList();

  return maplist.size();
}

extern "C" unsigned int
ob_SSS_SMARTS_native (const char *smarts_pattern, char *smiles)
{
  OBMol mol;
  OBConversion conv;
  OBSmartsPattern sp;
  string tmpStr (smiles);
  istringstream molstream (tmpStr);
  vector<vector<int> > maplist;

  conv.SetInAndOutFormats ("SMI", "SMI");

  conv.Read (&mol, &molstream);
  
  sp.Init(smarts_pattern);
  
  sp.Match(mol);

  maplist = sp.GetUMapList();

  return maplist.size();
}
