/*
 * PostgreSQL type definitions for uniqueidentifiers.
 *
 * $Id: $
 *
 * Copyright (C) 2001 Dmitry G. Mastrukov.
 * Code for _uid_compare is based on libuuid 
 *					Copyright (C) 1996, 1997 Theodore Ts'o.
 *
 * %Begin-Header%
 * This file may be redistributed under the terms of the GNU 
 * Library General Public License.
 * %End-Header%
 *
 */

#include <string.h>
#include <uuid/uuid.h>
#include "postgres.h"
#include "fmgr.h"


/*
 *	Here is the internal storage format for uniqueidentifiers.
 */

typedef struct uniqueidentifier
{
	uuid_t		uid;
}			uniqueidentifier;


/*
 *	Various forward declarations:
 */

Datum		uniqueidentifier_in(PG_FUNCTION_ARGS);
Datum		uniqueidentifier_out(PG_FUNCTION_ARGS);

Datum		uniqueidentifier_lt(PG_FUNCTION_ARGS);
Datum		uniqueidentifier_le(PG_FUNCTION_ARGS);
Datum		uniqueidentifier_eq(PG_FUNCTION_ARGS);
Datum		uniqueidentifier_ge(PG_FUNCTION_ARGS);
Datum		uniqueidentifier_gt(PG_FUNCTION_ARGS);
Datum		uniqueidentifier_ne(PG_FUNCTION_ARGS);

Datum		uniqueidentifier_cmp(PG_FUNCTION_ARGS);

Datum		newid(PG_FUNCTION_ARGS);
Datum           uniqueidentifier_text(PG_FUNCTION_ARGS);

static int32	_uid_compare(uniqueidentifier *u1, uniqueidentifier *u2);

/*
 *	uniqueidentifier input function:
 */

PG_FUNCTION_INFO_V1(uniqueidentifier_in);
Datum
uniqueidentifier_in(PG_FUNCTION_ARGS)
{
	uniqueidentifier *result;
	char *in = PG_GETARG_CSTRING(0);
	
	if (strlen(in) != 36)
	{
		elog(ERROR, "uniqueidentifier_in: invalid uniqueidentifier \"%s\"", in);
		PG_RETURN_POINTER (NULL);
	}

	result = (uniqueidentifier *) palloc(sizeof(uniqueidentifier));

	if (uuid_parse(in, result->uid) != 0)
	{
		elog(ERROR, "uniqueidentifier_in: wrong uniqueidentifier \"%s\"", in);
		PG_RETURN_POINTER (NULL);
	}
	PG_RETURN_POINTER (result);
}



/*
 *	uniqueidentifier output function:
 */

PG_FUNCTION_INFO_V1(uniqueidentifier_out);
Datum
uniqueidentifier_out(PG_FUNCTION_ARGS)
{
	char *result;
	uniqueidentifier *in = (uniqueidentifier *) PG_GETARG_POINTER(0);

	if (in == NULL)
		PG_RETURN_CSTRING (NULL);

	result = (char *) palloc(37);

	result[0] = '\0';

	uuid_unparse(in->uid, result);

	PG_RETURN_CSTRING (result);
}


/*
 *	uniqueidentifier to text conversion function:
 */

PG_FUNCTION_INFO_V1(uniqueidentifier_text);
Datum
uniqueidentifier_text(PG_FUNCTION_ARGS)
{
        char *s;
	uniqueidentifier *val = (uniqueidentifier *) PG_GETARG_POINTER(0);
	int len;
	text *result;

	s = (char *) palloc(37);
	s[0] = '\0';
	uuid_unparse(val->uid, s);

	len = strlen(s);

	result = (text *) palloc(VARHDRSZ + len);
	VARATT_SIZEP(result) = len + VARHDRSZ;
	memcpy(VARDATA(result),s,len);
	pfree(s);

	PG_RETURN_TEXT_P(result);
}


/*
 *	Boolean tests:
 */

PG_FUNCTION_INFO_V1(uniqueidentifier_lt);
Datum
uniqueidentifier_lt(PG_FUNCTION_ARGS)
{
	uniqueidentifier *u1 = (uniqueidentifier *) PG_GETARG_POINTER(0);
	uniqueidentifier *u2 = (uniqueidentifier *) PG_GETARG_POINTER(1);
	PG_RETURN_BOOL (_uid_compare(u1, u2) < 0);
};

PG_FUNCTION_INFO_V1(uniqueidentifier_le);
Datum
uniqueidentifier_le(PG_FUNCTION_ARGS)
{
	uniqueidentifier *u1 = (uniqueidentifier *) PG_GETARG_POINTER(0);
	uniqueidentifier *u2 = (uniqueidentifier *) PG_GETARG_POINTER(1);
	PG_RETURN_BOOL (_uid_compare(u1, u2) <= 0);
};

PG_FUNCTION_INFO_V1(uniqueidentifier_eq);
Datum
uniqueidentifier_eq(PG_FUNCTION_ARGS)
{
	uniqueidentifier *u1 = (uniqueidentifier *) PG_GETARG_POINTER(0);
	uniqueidentifier *u2 = (uniqueidentifier *) PG_GETARG_POINTER(1);
	
	if ((u1 == NULL) || (u2 == NULL)) {
	    if ((u1 == NULL) && (u2 == NULL)) 
		PG_RETURN_BOOL (true);
	    else
		PG_RETURN_BOOL (false);
	}		
	    
	if (memcmp(u1->uid, u2->uid, 16) == 0)
	    PG_RETURN_BOOL (true);
	else
	    PG_RETURN_BOOL (false);    
};

PG_FUNCTION_INFO_V1(uniqueidentifier_ge);
Datum
uniqueidentifier_ge(PG_FUNCTION_ARGS)
{
	uniqueidentifier *u1 = (uniqueidentifier *) PG_GETARG_POINTER(0);
	uniqueidentifier *u2 = (uniqueidentifier *) PG_GETARG_POINTER(1);
	PG_RETURN_BOOL (_uid_compare(u1, u2) >= 0);
};

PG_FUNCTION_INFO_V1(uniqueidentifier_gt);
Datum
uniqueidentifier_gt(PG_FUNCTION_ARGS)
{
	uniqueidentifier *u1 = (uniqueidentifier *) PG_GETARG_POINTER(0);
	uniqueidentifier *u2 = (uniqueidentifier *) PG_GETARG_POINTER(1);
	PG_RETURN_BOOL (_uid_compare(u1, u2) > 0);
};

PG_FUNCTION_INFO_V1(uniqueidentifier_ne);
Datum
uniqueidentifier_ne(PG_FUNCTION_ARGS)
{
	uniqueidentifier *u1 = (uniqueidentifier *) PG_GETARG_POINTER(0);
	uniqueidentifier *u2 = (uniqueidentifier *) PG_GETARG_POINTER(1);
	if ((u1 == NULL) || (u2 == NULL)) {
	    if ((u1 == NULL) && (u2 == NULL)) 
		PG_RETURN_BOOL (false);
	    else
		PG_RETURN_BOOL (true);
	}		
	if (memcmp(u1->uid, u2->uid, 16) == 0)
	    PG_RETURN_BOOL (false);
	else
	    PG_RETURN_BOOL (true);    
};


/*
 *	Comparison function for B-tree sorting:
 */

PG_FUNCTION_INFO_V1(uniqueidentifier_cmp);
Datum
uniqueidentifier_cmp(PG_FUNCTION_ARGS)
{
	uniqueidentifier *u1 = (uniqueidentifier *) PG_GETARG_POINTER(0);
	uniqueidentifier *u2 = (uniqueidentifier *) PG_GETARG_POINTER(1);
	PG_RETURN_INT32 (_uid_compare(u1, u2));
};

/*
 *	New uniqueidentifier:
 */

PG_FUNCTION_INFO_V1(newid);
Datum
newid(PG_FUNCTION_ARGS)
{
	uniqueidentifier *result = (uniqueidentifier *) palloc(sizeof(uniqueidentifier));

	uuid_generate(result->uid);
	
	PG_RETURN_POINTER(result);    
};


/*
 *	Internal compare function:
 */

#define UUCMP(u1,u2) if (u1 != u2) return((u1 < u2) ? -1 : 1);

static int32
_uid_compare(uniqueidentifier *u1, uniqueidentifier *u2)
{
	uint8	*ptr1 = u1->uid;
	uint32 time_low1;
	uint16 time_mid1;
	uint16 time_hi_and_version1;
	uint16 clock_seq1;
	uint8	*ptr2 = u2->uid;
	uint32 time_low2;
	uint16 time_mid2;
	uint16 time_hi_and_version2;
	uint16 clock_seq2;
	
	if ((u1 == NULL) || (u2 == NULL)) {
	    if ((u1 == NULL) && (u2 == NULL)) 
		return 0;
	    if (u1 == NULL)
		return 1;
	    else
		return -1;	
	}		

	if(memcmp(u1->uid, u2->uid, 16) == 0)
	    return 0;

	time_low1 = *ptr1++;
	time_low1 = (time_low1 << 8) | *ptr1++;
	time_low1 = (time_low1 << 8) | *ptr1++;
	time_low1 = (time_low1 << 8) | *ptr1++;
	time_low2 = *ptr2++;
	time_low2 = (time_low2 << 8) | *ptr2++;
	time_low2 = (time_low2 << 8) | *ptr2++;
	time_low2 = (time_low2 << 8) | *ptr2++;
	UUCMP(time_low1, time_low2);

	time_mid1 = *ptr1++;
	time_mid1 = (time_mid1 << 8) | *ptr1++;
	time_mid2 = *ptr2++;
	time_mid2 = (time_mid2 << 8) | *ptr2++;
	UUCMP(time_mid1, time_mid2);
	
	time_hi_and_version1 = *ptr1++;
	time_hi_and_version1 = (time_hi_and_version1 << 8) | *ptr1++;
	time_hi_and_version2 = *ptr2++;
	time_hi_and_version2 = (time_hi_and_version2 << 8) | *ptr2++;
	UUCMP(time_hi_and_version1, time_hi_and_version2);


	clock_seq1 = *ptr1++;
	clock_seq1 = (clock_seq1 << 8) | *ptr1++;
	clock_seq2 = *ptr2++;
	clock_seq2 = (clock_seq2 << 8) | *ptr2++;
	UUCMP(clock_seq1, clock_seq2);


	return memcmp(ptr1, ptr2, 6);
}

