/* $PostgresPy: if/src/tup.c,v 1.13 2004/07/27 16:17:34 flaw Exp $
 * 
 * † Instrument:
 *     Copyright 2004, rhid development. All Rights Reserved.
 *     
 *     Usage of the works is permitted provided that this
 *     instrument is retained with the works, so that any entity
 *     that uses the works is notified of this instrument.
 *     
 *     DISCLAIMER: THE WORKS ARE WITHOUT WARRANTY.
 *     
 *     [2004, Fair License; rhid.com/fair]
 *     
 * Description:
 *     PyPgTuple[_Type] implementation
 */
#include <setjmp.h>
#include <pputils.h>

#include <postgres.h>
#include <tcop/tcopprot.h>
#include <access/htup.h>
#include <access/hio.h>
#include <access/heapam.h>
#include <access/tupdesc.h>
#include <catalog/pg_type.h>
#include <parser/parse_type.h>
#include <utils/array.h>
#include <utils/syscache.h>
#include <utils/datum.h>
#include <pg.h>
#include <PGExcept.h>

#include <Python.h>
#include <structmember.h>
#include <py.h>

#include "module.h"
#include "datum.h"
#include "tup.h"
#include "tupd.h"
#include "type.h"
#include "obj.h"
#include "utils.h"

static int
off_from_pyobj(PyObj self, PyObj ind)
{
	int off;

	if (PyInt_CheckExact(ind))
		off = PYASINT(ind);
	else if (PyLong_CheckExact(ind))
		off = PyLong_AsLong(ind);
	else
		off = PgTupD_AttOffset(PgTupD(PgTup_FetchTDO(self)), ind);
	
	return(off);
}

static long
tup_length(PgTup self)
{
	return(PgTup_FetchNatts(self));
}

static PyObj
tup_item(PgTup self, int attnum)
{
	bool isnull = false;
	TupleDesc td;
	int natts;
	HeapTuple ht;
	Datum datum;
	Form_pg_type ts;
	PyObj rob = NULL;

	td = PgTup_FetchTDOTD(self);
	natts = PgTup_FetchNatts(self);

	if (attnum > natts)
	{
		PyErr_Format(PyExc_IndexError,
				"index(%d) out of range(%d)", attnum, natts);
		return(NULL);
	}

	ht = PgTup_FetchHT(self);
	datum = HeapTuple_FetchAttribute(ht, attnum, td, &isnull);
	if (PyErr_Occurred())
		return(NULL);

	ts = PGOTYPESTRUCT(rob);
	if (ts->typrelid)
	{
		HeapTuple tup;
		PyObj tdo;
		tdo = PgTupD_FromRelationOid(ts->typrelid);
		if (tdo == NULL)
			return(NULL);

		tup = heap_copytuple((HeapTuple) datum);
		rob = PyPgTuple_New(tup, tdo);
	}
	else
	{
		rob = PgObj_FromDatumAndTypeOid(datum, td->attrs[attnum]->atttypid);
		if (!ts->typbyval)
			PgObj(rob)->ob_datum = datumCopy(datum, false, ts->typlen);
		
		PgObjFlags(rob) |= isnull ? PgObjFlag_Null : PgObjFlag_Void;
	}
	return(rob);
}

static PyObj
tup_slice(PgTup self, int from, int to)
{
	PyObj rob, v;
	int c;

	rob = PyTuple_New(to - from);
	for (c = from; c < to; ++c)
	{
		v = tup_item(self, c);
		PyTuple_SET_ITEM(rob, c - from, v);
	}

	return(rob);
}

static PyObj
tup_ass_item(PgTup self, int att, PyObj ni)
{
	RETURN_NONE;
}

static PyObj
tup_ass_slice(PgTup self, int from, int to, PyObj ni)
{
	RETURN_NONE;
}

static int
tup_contains(PgTup self, PyObj ob)
{
	return(0);
}

static PySequenceMethods PyPgTupleAsSequence = {
	(inquiry)tup_length,						/* sq_length */
	(binaryfunc)NULL,							/* sq_concat */
	(intargfunc)NULL,							/* sq_repeat */
	(intargfunc)tup_item,					/* sq_item */
	(intintargfunc)tup_slice,				/* sq_slice */
	(intobjargproc)tup_ass_item,			/* sq_ass_item */
	(intintobjargproc)tup_ass_slice,		/* sq_ass_slice */
	(objobjproc)tup_contains,				/* sq_contains */
	(binaryfunc)NULL,							/* sq_inplace_concat */
	(intargfunc)NULL,							/* sq_inplace_repeat */
};

static PyObj
tup_subscript(PgTup self, PyObj sub)
{
	PyObj rob;
	rob = tup_item(self, off_from_pyobj((PyObj)self, sub));
	return(rob);
}

static PyObj
tup_ass_subscript(PgTup self, PyObj from, PyObj to, PyObj with)
{
	PyObj rob;
	rob = tup_ass_slice(self,
		off_from_pyobj((PyObj)self, from),
		off_from_pyobj((PyObj)self, to),
		with
	);
	return(rob);
}

static PyMappingMethods PyPgTupleAsMapping = {
	(inquiry)tup_length,						/* mp_length */
	(binaryfunc)tup_subscript,				/* mp_subscript */
	(objobjargproc)tup_ass_subscript,	/* mp_ass_subscript */
};

static PyNumberMethods PyPgTupleAsNumber = {
	NULL,
};

static PyMethodDef PyPgTuple_Methods[] = {
	/* {"name", (PyCFunction)FunctionRef, flags, "docstring"}, */
	{NULL}
};

static void
tup_dealloc(PgTup self)
{
	heap_freetuple(PgTup_FetchHT(self));
	XDECREF(self->tup_desc);
	PyObject_Del((PyObj)self);
}

static PyObj
tup_getattr(PgTup self, char *attr)
{
	PyObj rob = NULL;

	if (!strcmp(attr, "desc"))
	{
		rob = self->tup_desc;
		INCREF(rob);
	}
	else if (!strcmp(attr, "oid"))
	{
		rob = PgObj_FromDatumAndTypeOid(PgTup_FetchOid(self), OIDOID);
	}
	else if (!strcmp(attr, "tableoid"))
	{
		rob = PgObj_FromDatumAndTypeOid(PgTup_FetchTableOid(self), OIDOID);
	}
	else if (!strcmp(attr, "Xmin"))
	{
		rob = PgObj_FromDatumAndTypeOid(PgTup_FetchXmin(self), XIDOID);
	}
	else if (!strcmp(attr, "Xmax"))
	{
		rob = PgObj_FromDatumAndTypeOid(PgTup_FetchXmax(self), XIDOID);
	}
	else if (!strcmp(attr, "Xvac"))
	{
		rob = PgObj_FromDatumAndTypeOid(PgTup_FetchXvac(self), XIDOID);
	}
	else if (!strcmp(attr, "Cmin"))
	{
		rob = PgObj_FromDatumAndTypeOid(PgTup_FetchCmin(self), CIDOID);
	}
	else if (!strcmp(attr, "Cmax"))
	{
		rob = PgObj_FromDatumAndTypeOid(PgTup_FetchCmax(self), CIDOID);
	}
	else if (!strcmp(attr, "TID"))
	{
		ItemPointer to, from;
		from = &PgTup_FetchTID(self);
		to = palloc(SizeOfIptrData);
		if (to)
		{
			ItemPointerCopy(from, to);
			rob = PgObj_FromDatumAndTypeOid(PointerGetDatum(to), TIDOID);
		}
	}
	else
	{
		PyErr_Format(PyExc_AttributeError,
			"Postgres.Tuple objects have no attribute '%s'",
			attr
		);
	}

	return(rob);
}

static int
tup_compare(PgTup self, PgTup that)
{
	unsigned long i;
	TupleDesc selftupd, thattupd;

	selftupd = PgTup_FetchTDOTD(self);
	thattupd = PgTup_FetchTDOTD(that);

	if (selftupd->natts != thattupd->natts)
		return(-1);

	for (i = 0; i < selftupd->natts; ++i)
		if ((selftupd->attrs[i])->atttypid != (thattupd->attrs[i])->atttypid)
			return(-1);

	return(-1);
}

static PyObj
tup_repr(PgTup self)
{
	PyObj rob;
	HeapTuple ht;
	TupleDesc td;
	unsigned long i, natts;
	
	ht = PgTup_FetchHT(self);
	td = PgTup_FetchTDOTD(self);
	natts = td->natts;

	rob = PYSTR("(");

	for (i = 0; i < natts; ++i)
	{
		bool isnull;
		Datum d;
		HeapTuple ttup;
		Form_pg_type typs;
		Form_pg_attribute att;
		Oid atttypid;
		char *str = NULL;

		att = td->attrs[i];
		atttypid = att->atttypid;

		ttup = SearchSysCache(TYPEOID, atttypid, 0, 0, 0);
		typs = TYPESTRUCT(ttup);

		d = HeapTuple_FetchAttribute(ht, i, td, &isnull);
		if (PyErr_Occurred())
		{
			ReleaseSysCache(ttup);
			ttup = NULL;

			XDECREF(rob);
			rob = NULL;

			return(NULL);
		}

		if (isnull)
			PyString_ConcatAndDel(&rob, PYSTR(" NULL "));
		else
		{
			PyString_ConcatAndDel(&rob, PYSTR(" '"));
			str = (char *) OidFunctionCall1(typs->typoutput, d);
			PyString_ConcatAndDel(&rob, PYSTR(str));
			pfree(str);
			PyString_ConcatAndDel(&rob, PYSTR("' "));
		}

		str = NameStr(att->attname);
	
		if (strlen(str))
		{
			PyString_ConcatAndDel(&rob, PYSTR(str));
			PyString_ConcatAndDel(&rob, PYSTR(" "));
		}
		
		str = NameStr(typs->typname);
		PyString_ConcatAndDel(&rob, PYSTR(str));

		ReleaseSysCache(ttup);

		if (i+1 < natts)
			PyString_ConcatAndDel(&rob, PYSTR(","));
	}

	PyString_ConcatAndDel(&rob, PYSTR(" )"));
	return(rob);
}

static PyObj
tup_str(PgTup self)
{
	PyObj rob;
	HeapTuple ht;
	TupleDesc td;
	unsigned long i, natts;
	
	ht = PgTup_FetchHT(self);
	td = PgTup_FetchTDOTD(self);
	natts = td->natts;

	rob = PYSTR("");

	for (i = 0; i < natts; ++i)
	{
		bool isnull;
		Datum d;
		HeapTuple ttup;
		Form_pg_type typs;
		Form_pg_attribute att;
		Oid atttypid;
		char *str = NULL;

		att = td->attrs[i];
		atttypid = att->atttypid;

		ttup = SearchSysCache(TYPEOID, atttypid, 0, 0, 0);
		typs = TYPESTRUCT(ttup);

		d = HeapTuple_FetchAttribute(ht, i, td, &isnull);
		if (PyErr_Occurred())
		{
			ReleaseSysCache(ttup);
			return(NULL);
		}
		if (isnull)
			PyString_ConcatAndDel(&rob, PYSTR(" NULL "));
		else
		{
			PyString_ConcatAndDel(&rob, PYSTR(" '"));
			str = (char *) OidFunctionCall1(typs->typoutput, d);
			PyString_ConcatAndDel(&rob, PYSTR(str));
			pfree(str);
			PyString_ConcatAndDel(&rob, PYSTR("'::"));
		}

		str = NameStr(typs->typname);
		PyString_ConcatAndDel(&rob, PYSTR(str));

		ReleaseSysCache(ttup);

		if (i+1 < natts)
			PyString_ConcatAndDel(&rob, PYSTR(","));
	}

	return(rob);
}

const char PyPgTuple_Doc[] = "Postgres HeapTuple interface type";
PyTypeObject PyPgTuple_Type = {
	PyObject_HEAD_INIT(NULL)
	0,										/* ob_size */
	"Postgres.Tuple",					/* tp_name */
	sizeof(PyPgTuple),				/* tp_basicsize */
	0,										/* tp_itemsize */
	(destructor)tup_dealloc,		/* tp_dealloc */
	NULL,									/* tp_print */
	(getattrfunc)tup_getattr,		/* tp_getattr */
	(setattrfunc)NULL,				/* tp_setattr */
	(cmpfunc)tup_compare,			/* tp_compare */
	(reprfunc)tup_repr,				/* tp_repr */
	&PyPgTupleAsNumber,				/* tp_as_number */
	&PyPgTupleAsSequence,			/* tp_as_sequence */
	&PyPgTupleAsMapping,				/* tp_as_mapping */
	(hashfunc)NULL,					/* tp_hash */
	NULL,									/* tp_call */
	(reprfunc)tup_str,				/* tp_str */
	NULL,									/* tp_getattro */
	NULL,									/* tp_setattro */
	NULL,									/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT,			   /* tp_flags */
	(char*)PyPgTuple_Doc,			/* tp_doc */
	(traverseproc)NULL,				/* tp_traverse */
	(inquiry)NULL,						/* tp_clear */
	(richcmpfunc)NULL,				/* tp_richcompare */
	(long)0,								/* tp_weaklistoffset */
	(getiterfunc)PySeqIter_New,	/* tp_iter */
	(iternextfunc)NULL,				/* tp_iternext */
	PyPgTuple_Methods,				/* tp_methods */
	NULL,									/* tp_members */
	NULL,									/* tp_getset */
	&PyPgDatum_Type,					/* tp_base */
	NULL,									/* tp_dict */
	NULL,									/* tp_descr_get */
	NULL,									/* tp_descr_set */
	NULL,									/* tp_dictoffset */
	(initproc)NULL,					/* tp_init */
	NULL,									/* tp_alloc */
	PyType_GenericNew,				/* tp_new */
};

PyObj
PyPgTuple_New(HeapTuple tup, PyObj tupd)
{
	PgTup self;
	AssertArg(tup != NULL);
	AssertArg(tupd != NULL);

	self = PgTup_NEW();
	if (self == NULL)
		return(NULL);

	self->ob_refcnt = 1;
	self->ob_datum_flags = PgDatFlag_Ref;
	self->ob_datum = PointerGetDatum(tup);
	self->tup_desc = tupd;
	INCREF(tupd);

	return((PyObj)self);
}

PyObj
PyPgTypeTuple_FromPyObject(PyObj ob)
{
	Oid typ;

	typ = TypeOid_FromPyObject(ob);
	return(PyPgSearchSysCache(TYPEOID, (Datum)typ, 0, 0, 0));
}

/*
 * PyPgSearchSysCache
 *
 * Equivalent to SearchSysCache, but returns a PyObj instead of a HeapTuple.
 */
PyObj
PyPgSearchSysCache(int cacheId, Datum k1, Datum k2, Datum k3, Datum k4)
{
	HeapTuple htup = NULL;
	Oid httoid = 0;
	PyObj to, tdo = NULL;

	htup = SearchSysCacheCopy(cacheId, k1, k2, k3, k4);
	if (htup == NULL)
	{
		PyErr_Format(PyExc_PgErr,
			"SearchSysCache Failure(%d,%lu,%lu,%lu,%lu)",
			cacheId, k1, k2, k3, k4);
		return(NULL);
	}

	httoid = HeapTuple_FetchTableOid(htup);

	tdo = PyPgTupleDesc_FromRelationOid(httoid);
	to = PyPgTuple_New(htup, tdo);

	return(to);
}

/*
 * HeapTuple_FromIterableAndTupleDesc
 *
 * Convert a Python iterable to a HeapTuple
 * Attempt to be as flexible as possible by using Datum_From...
 */
HeapTuple
HeapTuple_FromIterableAndTupleDesc(PyObj so, TupleDesc td)
{
	PyObj iter = NULL;
	PyObj attr = NULL;
	
	unsigned int i = 0;
	char *nulls = NULL;
	Datum *values = NULL;
	HeapTuple ht = NULL;

	nulls = palloc(sizeof(char) * td->natts);
	values = palloc(sizeof(Datum) * td->natts);

	iter = PyObject_GetIter(so);
	while ((attr = PyIter_Next(iter)))
	{
		/* Ignore the overflow */
		if (i >= td->natts)
			break;

		if (attr == Py_None ||( PgObj_TypeCheck(attr) && PgObj_IsNull(attr)))
		{
			nulls[i] = 'n';
			values[i] = 0;
		}
		else
		{
			nulls[i] = ' ';
			values[i] = Datum_FromPyObjectAndTypeOid(attr, td->attrs[i]->atttypid);
		}

		DECREF(attr);
		++i;
	}
	DECREF(iter);
	
	/* Handle underflow by NULLing out the rest */
	if (i < td->natts)
		for (; i < td->natts; ++i)
		{
			nulls[i] = 'n';
			values[i] = 0;
		}

	ht = heap_formtuple(td, values, nulls);

	while (i-- != 0)
	{
		if (!(td->attrs[i]->attbyval) && nulls[i] != 'n')
			pfree(DatumGetPointer(values[i]));
	}
	pfree(values);
	pfree(nulls);

	return(ht);
}
