# $Id: HTML.pm,v 1.28 2004/02/07 00:12:22 mig Exp $
######################################
# Comas - Conference Management System
######################################
# Copyright 2003 CONSOL
# Congreso Nacional de Software Libre (http://www.consol.org.mx/)
#   Gunnar Wolf <gwolf@gwolf.cx>
#   Manuel Rabade <mig@mig-29.net>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
######################################

######################################
# Module: Comas::HTML
# Manages Comas' HTML front-end generator
######################################
# Depends on:

# Something to interface with a Web server (a mod_perl module, a CGI
# script, or some other program interfacing with the users' requests
# Comas::DB - Handles the database connectivity for a Comas module
# Comas::HTML::common - Common Functions for the HTML Front End
# Comas::HTML::attendees - Manage attendees registration

use strict;
use warnings;

package Comas::HTML;

use Switch;

use Comas::Conf qw(:read);
use Comas::HTML::attendees qw(:all);
use Comas::HTML::common qw(:all);
use Comas::HTML::authors qw(:all);
use Comas::HTML::general qw(:all);
use Comas::HTML::committee qw(:all);
use Comas::HTML::committee::academic_committee qw(:all);
use Comas::HTML::committee::admin_mgr qw(:all);
use Comas::HTML::committee::payment qw(:all);
use Comas::HTML::committee::common qw(:all);

=head1 NAME

Comas::HTML - Comas' HTML front-end generator

=head1 SYNOPSIS

  $html = Comas::HTML->new($html = Comas::HTML->new(-db => $db,
                           -db_adm => $db_adm,
                           -path => \@path,
                           -params => $params,
                           -session => \%session,
                           -upload => $upload);

  $ret = $html->process;

  $debug = $html->debug_object;

  %http_headers = $html->http_headers;

=head1 REQUIRES



=head1 EXPORTS



=head1 SEE ALSO



=head1 AUTHOR

Gunnar Wolf, gwolf@gwolf.cx

Manuel Rabade, mig@mig-29.net

Comas has been developed for CONSOL, Congreso Nacional de Software Libre,
http://www.consol.org.mx/

=head1 COPYRIGHT

Copyright 2003 Gunnar Wolf and Manuel Rabade

This library is free software, you can redistribute it and/or modify it
under the terms of the GPL version 2 or later.

=cut

################################################################################
# Parent Functions

sub new {
    my $class = shift;
    my $p = { @_ };

    # We create a reference to an array call to store all the configurationfor
    # for the front-end.
    
    ## Aqui puede haber mas chequeos, por ej, que la ruta de tmpl_path exista
    ## Ahi de tarea :-P y tambien mas opciones de configuracion :-D
    
    $p->{-config} = { -tmpl_path => get_conf($p->{-db},'tmpl_path'), 
                      -prop_file_path => get_conf($p->{-db},'prop_files_path') };

    foreach my $config (keys %{$p->{-config}}) {
        unless ( defined $p->{-config}->{$config}) {
            return undef;
        }
    }

    # Let's normalize the paths, to end with a '/'

    foreach my $config (keys %{$p->{-config}}) {
        if (index($config, 'path') != -1) {
            if (substr($p->{-config}->{$config}, -1) ne '/') {
                $p->{-config}->{$config} .= '/';
            }
        }
    }
    
    bless ($p, $class);
    return $p;
}

sub http_headers {
    my $h = shift;
    if (ref($h->{-http_headers}))  {
        return %{$h->{-http_headers}};
    } else {
        return undef;
    }
}

sub process {
    my $p = shift;
    if (defined $p->{-path}->[0]) {
        if ($p->{-path}->[0] eq 'attendees') {
            return $p->attendees;
        } elsif ($p->{-path}->[0] eq 'general') {
            return $p->general;
        } elsif ($p->{-path}->[0] eq 'committee') {
            return $p->committee;
        } else {
            return $p->main_menu;
        }
    } else {
        return $p->main_menu;
    }	
}

sub debug_object {
    my $p = shift;
    my $ret = "<hr>";
    
    foreach my $test (keys %$p) {
        $ret .= $test . " - ";
        $ret .= ref($p->{$test}) . "<br>";
    }
    
    $ret .= "<br> PARMAS: <br>";
    foreach my $test1 (keys %{$p->{-params}}) {
        $ret .= "$test1 - ". $p->{-params}->{$test1} ."<br>";
    }
    
    $ret .= "<br> PATH: <br>";
    foreach my $test2 (@{$p->{-path}}) {
        $ret .= $test2 . "<br>";
    }
    
    $ret .= "<br> SESSION: <br>";
    foreach my $test3 (keys %{$p->{-session}}) {
        $ret .= "$test3 - ". $p->{-session}->{$test3} ."<br>";
    }
    
    $ret .= "<br> CONFIG: <br>";
    foreach my $test4 (keys %{$p->{-config}}) {
        $ret .= "$test4 - ". $p->{-config}->{$test4} ."<br>";
    }

    $ret .= "<br> HTTP HEADERS: <br>";
    foreach my $test5 (keys %{$p->{-http_headers}}) {
        $ret .= "$test5 - ". $p->{-http_headers}->{$test5} ."<br>";
    }
    
    return $ret;
}


###############################################################################
# All the logic starts here

sub attendees {
    my $h = shift;

    switch ($h->{-path}->[1]) {
        case 'account' {
            # If the path is account and the id of the person is defined in the
            # session, let's show the attendees menu or any section.
            if(defined $h->{-session}->{-person}) {
                switch ($h->{-path}->[2]) {
                    case 'data' { return $h->attendees_data; }
                    case 'logout' { return $h->attendees_logout; }
                    case 'passwd' {  return $h->attendees_passwd; }
                    case 'authors' {
                        # We got an author !
                        switch ($h->{-path}->[3]) {
                            case 'data' { return $h->authors_data; }
                            case 'data_delete' {
                                return $h->authors_data_delete; }
                            else { return $h->attendees_menu; }
                        }
                    }
                    case 'proposals' {
                        # The proposals stuff ...
                        switch($h->{-path}->[3]) {
                            case 'new' { return $h->proposals_new; }
                            case 'list' { return $h->proposals_list; }
                            case 'edit' { return $h->proposals_edit; }
                            case 'delete' { return $h->proposals_delete; }
                            case 'authors' { return $h->proposals_authors; }
                            case 'file' { return $h->proposals_file; }
                            else { return $h->attendees_menu; }
                        }
                    }
                    else { return $h->attendees_menu; }
                }
            } else {
                # If the ID isn't in the session, let's show the login form.
                return $h->attendees_login;
            }
        }
        case 'new' { return $h->attendees_new; }
        case 'recover_passwd' { return $h->attendees_recover_passwd; }
        else { return $h->main_menu; }
    }
}

###############################################################################

sub general {
    my $h = shift;

    switch ($h->{-path}->[1]) {
        case 'proposals' {
            if (defined $h->{-path}->[2]) {
                if ($h->{-path}->[3] eq 'file') {
                    return $h->proposals_get_file;
                } else {
                    return $h->proposals_details;
                }
            } else {
                return $h->proposals_general_list;
            }
        } else {
            return $h->main_menu;
        }
    }
}

###############################################################################

sub committee {
    my $h = shift;
    if(defined $h->{-session}->{-admin}) {
        switch ($h->{-path}->[1]) {
            case 'passwd' { return $h->committee_passwd; }
            case 'logout' { return $h->committee_logout; }
            case 'academic_committee' {
                if (defined $h->{-path}->[2]) {
                    return $h->academic_committee_prop_revision;
                } else {
                    return $h->academic_committee_prop_list;
                }   
            }
            case 'admin_mgr' {
                if (defined $h->{-path}->[2]) {
                    return $h->admin_mgr_edit;
                } else {
                    return $h->admin_mgr_menu;
                }
            }
            case 'payment_reg' {
                if(defined $h->{-path}->[2]) { 
                    if ($h->{-path}->[2] eq 'search') {
                        return $h->payment_reg_search_person;
                    } elsif ($h->{-path}->[2] eq 'select') {
                        return $h->payment_reg_select;
                    } elsif ($h->{-path}->[2] eq 'confirm') {
                        return $h->payment_reg_confirm;
                    } elsif ($h->{-path}->[2] eq 'register') {
                        return $h->payment_reg_register;
                    } elsif ($h->{-path}->[2] eq 'print_sticker') {
                        return $h->payment_print_sticker;
                    } elsif ($h->{-path}->[2] eq 'print_diploma') {
                        return $h->payment_print_diploma;
                    } else {
                        return $h->payment_reg_select_person;
                    }
                } else {
                    return $h->payment_reg_select_person;
                }
            }
            case 'payment_mod' {
                if(defined $h->{-path}->[2]) { 
                    if ($h->{-path}->[2] eq 'search') {
                        return $h->payment_mod_search_person_delete;
                    } elsif ($h->{-path}->[2] eq 'select') {
                        return $h->payment_mod_select_delete;
                    } elsif ($h->{-path}->[2] eq 'confirm') {
                        return $h->payment_mod_confirm_delete;
                    } elsif ($h->{-path}->[2] eq 'delete') {
                        return $h->payment_mod_delete;
                    } else {
                        return $h->payment_mod_select_person_delete;
                    }
                } else {
                    return $h->payment_mod_select_person_delete;
                }
            }
            else { return $h->committee_menu; }
        }
    } else {
        # If the ID isn't in the session, let's show the login form.
        return $h->committee_login;
    }
}

1;


# $Log: HTML.pm,v $
# Revision 1.28  2004/02/07 00:12:22  mig
# - Lista la consulta y eliminacion de pagos
#
# Revision 1.27  2004/02/06 02:02:51  mig
# - Lista la impresion de constancias :)
#
# Revision 1.26  2004/02/05 20:16:16  mig
# - Al parecer listo los gafetes :)
#
# Revision 1.25  2004/02/03 19:35:17  mig
# - Lista la recuperacion de contraseas
#
# Revision 1.24  2004/01/24 01:43:32  mig
# - Lista la busqueda de personas :)
#
# Revision 1.23  2004/01/23 07:46:33  mig
# - Agrego la funcion para registrar pagos.
#
# Revision 1.22  2004/01/20 17:50:16  mig
# - Los casos de pagos
#
# Revision 1.21  2004/01/11 04:54:56  mig
# - Un poco de limpieza.
#
# Revision 1.20  2003/12/21 04:29:13  mig
# - Cuando el constructor del objeto carga los paths los normaliza a que
#   todos terminen con '/'.
#
# Revision 1.19  2003/12/20 04:14:51  mig
# - Agrego tags Id y Log que expanda el CVS
#
