# $Id: academic_committee.pm,v 1.4 2004/01/05 19:36:32 gwolf Exp $
######################################
# Comas - Conference Management System
######################################
# Copyright 2003 CONSOL
# Congreso Nacional de Software Libre (http://www.consol.org.mx/)
#   Gunnar Wolf <gwolf@gwolf.cx>
#   Manuel Rabade <mig@mig-29.net>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
######################################

######################################
# Module: Comas::Admin::academic_committee
# Setting status and scheduling of proposals
######################################
# Depends on:
#
# Comas::Common - Common functions for various Comas modules
# Comas::Proposal - Handles the interaction with a proposal for Comas
# Comas::Schedule - Creating and modifying the schedule for the accepted 
#     proposals

# This module is not meant to be called on its own, it should be called from
# Comas::Admin.
package Comas::Admin::academic_committee;

use strict;
use warnings;
use Carp;
use Comas::Common qw(valid_hash);
use Comas::Schedule;
use Comas::Proposal;
use Exporter;
our @ISA = qw(Exporter);
our @EXPORT_OK = qw(set_status set_comments set_track_id set_prop_type_id
		    schedule unschedule);
our %EXPORT_TAGS = (prop_modif => [qw(set_status set_comments set_track_id
				      set_prop_type_id)],
		    schedule => [qw(schedule unschedule)]);

=head1 NAME

Comas::Admin::academic_committee - Setting status and scheduling of proposals

=head1 SYNOPSIS

This module is not meant to be used by itself, but as an auxiliary to
L<Comas::Admin>. Please check L<Comas::Admin>'s documentation, in the
B<Proposal approval and scheduling> section, for further details on the usage 
for this functions.

The functions in this module are exported in two categories:

=over 4

=item * b<prop_modif>: Modify the proposal's data (set_status, set_comments,
set_track_id, set_prop_type_id)

=item * b<schedule>: Set, unset or modify the scheduling information for a
proposal (schedule, unschedule)

=back

=head1 SEE ALSO

L<Comas::Admin> module documentation

=head1 AUTHOR

Gunnar Wolf, gwolf@gwolf.cx
Manuel Rabade, mig@mig-29.net

Comas has been developed for CONSOL, Congreso Nacional de Software Libre,
http://www.consol.org.mx/

=head1 COPYRIGHT

Copyright 2003 Gunnar Wolf and Manuel Rabade

This library is free software, you can redistribute it and/or modify it
under the terms of the GPL version 2 or later.

=cut

sub set_status {
    my ($adm, $prop, $status, $sth);
    $adm = shift;
    $prop = shift;
    $status = shift;

    unless ($adm->ck_admin_task(-task=>'academic_committee')) {
	carp 'Access denied.';
	return undef;
    }

    unless ($sth = $adm->{-db}->prepare('UPDATE proposal SET prop_status_id = ?
            WHERE id = ?') and $sth->execute($status, $prop)) {
	carp 'Could not set proposal to the specified status';
	return undef;
    }

    return 1;
}

sub set_comments {
    my ($adm, $prop, $comments, $sth);
    $adm = shift;
    $prop = shift;
    $comments = shift;

    unless ($sth = $adm->{-db}->prepare('UPDATE proposal SET comments = ?
            WHERE id = ?') and $sth->execute($comments, $prop)) {
	carp 'Could not set the requested comments';
	return undef;
    }

    return 1;
}

sub set_track_id {
    my ($adm, $prop_id, $track, $prop);
    $adm = shift;
    $prop_id = shift;
    $track = shift;

    unless ($prop = Comas::Proposal->new(-db=>$adm->{-db}, -id=>$prop_id) and
	    $prop->set_track_id($track)) {
	carp 'Could not set the requested track ID';
	return undef;
    }
    return 1;
}

sub set_prop_type_id {
    my ($adm, $prop_id, $type, $prop);
    $adm = shift;
    $prop_id = shift;
    $type = shift;

    unless ($prop = Comas::Proposal->new(-db=>$adm->{-db}, -id=>$prop_id) and
	    $prop->set_prop_type_id($type)) {
	carp 'Could not set the requested proposal type ID';
	return undef;
    }
    return 1;
}

sub schedule {
    # Just a wrapper for Comas::Schedule's schedule method
    my ($adm, $sched);

    unless ($adm->ck_admin_task(-task=>'academic_committee')) {
	carp 'Access denied.';
	return undef;
    }

    $sched = Comas::Schedule->new($adm->{-db}) or return undef;

    return $sched->schedule(@_);
}

sub unschedule {
    # Just a wrapper for Comas::Schedule's unschedule method
    my ($adm, $prop, $sched);
    $adm = shift;
    $prop = shift;

    unless ($adm->ck_admin_task(-task=>'academic_committee')) {
	carp 'Access denied.';
	return undef;
    }

    $sched = Comas::Schedule->new($adm->{-db}) or return undef;

    return $sched->unschedule(-prop => $prop);
}

1;

# $Log: academic_committee.pm,v $
# Revision 1.4  2004/01/05 19:36:32  gwolf
# - Schedule.pm, Schedule/Room.pm: Por fin ya agendo sobre cualquier criterio!
#   (slo falta hacer que evite sobrecargar de ponencais con el mismo track un
#   mismo espacio)
# - Admin/academic_committee.pm: Agrego unas funcioncillas que pidi Mig
# - Admin.pm: Quito todas las referencias a db_param, que ya no usaremos
#
# Revision 1.3  2003/12/20 04:14:51  mig
# - Agrego tags Id y Log que expanda el CVS
#
