import json
import pathlib
import click

from sigma.cli.rules import load_rules
from sigma.analyze.attack import score_functions, calculate_attack_scores
from sigma.data.mitre_attack import mitre_attack_techniques_tactics_mapping, mitre_attack_version

@click.group(name="analyze", help="Analyze Sigma rule sets")
def analyze_group():
    pass

@analyze_group.command(
        name="attack",
        help="Create MITRE™️ ATT&CK heatmaps from Sigma rule set. Score functions are: " + ", ".join((
            f"{definition[1]} ({func})"
            for func, definition in score_functions.items()
        ))
    )
@click.option(
    "--file-pattern", "-P",
    default="*.yml",
    show_default=True,
    help="Pattern for file names to be included in recursion into directories.",
)
@click.option(
    "--subtechniques/--no-subtechniques", "-s/-S",
    default=True,
)
@click.option(
    "--max-color", "-c",
    default="#ff0000",
    show_default=True,
    help="Color used for maximum score."
)
@click.option(
    "--min-color", "-C",
    default="#ffffff00",
    show_default=True,
    help="Color used for zero score."
)
@click.option(
    "--max-score", "-m",
    type=int,
    default=None,
    help="Set fixed maximum score. All scores above are rendered as maximum. Increases color scale resolution for scores below.",
)
@click.option(
    "--min-score", "-M",
    type=int,
    default="0",
    show_default=True,
    help="Minimum score. All scores below are not explicitly colored.",
)
@click.argument(
    "function",
    type=click.Choice(score_functions.keys()),
)
@click.argument(
    "output",
    type=click.File("w"),
)
@click.argument(
    "input",
    nargs=-1,
    required=True,
    type=click.Path(exists=True, allow_dash=True, path_type=pathlib.Path),
)
def analyze_attack(file_pattern, subtechniques, max_color, min_color, max_score, min_score, function, output, input):
    rules = load_rules(input, file_pattern)
    score_function = score_functions[function][0]
    scores = calculate_attack_scores(rules, score_function, not subtechniques)
    layer_techniques = [
        {
            "techniqueID": technique,
			"tactic": tactic,
			"score": score,
			"color": "",
			"comment": "",
			"enabled": True,
			"metadata": [],
			"links": [],
			"showSubtechniques": False,
        }
        for technique, score in scores.items()
        for tactic in mitre_attack_techniques_tactics_mapping.get(technique, [])
    ]
    layer = {
        "name": "layer",
        "versions": {
            "attack": mitre_attack_version,
            "navigator": "4.8.1",
            "layer": "4.4"
        },
        "domain": "enterprise-attack",
        "description": f"Sigma coverage heatmap generated by Sigma CLI with score function {function}",
        "gradient": {
            "colors": [
                min_color,
                max_color,
            ],
            "minValue": min_score,
            "maxValue": max_score or max(scores.values())
        },
        "techniques": layer_techniques,
    }
    json.dump(layer, output, indent=2)