/*
 * xtc - The eXTensible Compiler
 * Copyright (C) 2006-2008 Robert Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301,
 * USA.
 */
package xtc.parser;

import java.util.List;

/**
 * Definition of node property names.
 *
 * @author Robert Grimm
 * @version $Revision: 1.17 $
 */
public class Properties {

  // Hide the constructor.
  private Properties() { /* Nothing to do. */ }

  /**
   * The consumer property.  It is associated with productions, has a
   * boolean value, and indicates whether a production may consume the
   * input.
   */
  public static final String CONSUMER = "consumer";

  /**
   * The cost property.  It is associated with productions, has an
   * integer value, and indicates a production's estimated cost for
   * inlining.
   */
  public static final String COST = "cost";

  /**
   * The duplicates property.  It is associated with productions, has
   * a list of strings value, and indicates the names of the original
   * productions folded into the annotated one.
   */
  public static final String DUPLICATES = "duplicates";

  /**
   * The empty property.  It is associated with productions, has a
   * boolean value, and indicates whether a production may match the
   * empty input.
   */
  public static final String EMPTY = "empty";

  /**
   * The generic property.  When associated with a production, it has
   * a {@link #GENERIC_NODE} value indicating that the production's
   * semantic value is a generic node with the component values as its
   * children or a {@link #GENERIC_RECURSION} value indicating that
   * the production's value is the result of a left-recursive
   * production.  When associated with a grammar, it has boolean value
   * indicating whether the grammar contains generic nodes.
   */
  public static final String GENERIC = "generic";

  /** The generic node value. */
  public static final String GENERIC_NODE = "node";

  /** The generic recursion value. */
  public static final String GENERIC_RECURSION = "recursion";

  /**
   * The formatting property.  It is associated with sequences, has a
   * list of bindings as its value, and indicates that a recursive
   * alternative in a directly left-recursive generic production has
   * formatting annotations for the node generated by the promise.
   */
  public static final String FORMATTING = "formatting";

  /**
   * The lexical property.  It is associated with productions, has a
   * boolean value, and indicates whether a production recognizes
   * lexical syntax.
   */
  public static final String LEXICAL = "lexical";

  /**
   * The locatable property.  It is associated with a grammar, has a
   * boolean value, and indicates whether the corresponding parser
   * uses the locatable interface.
   */
  public static final String LOCATABLE = "locatable";

  /**
   * The meta-data property.  It is associated with productions and
   * has a {@link MetaData} value containing a production's meta-data.
   */
  public static final String META_DATA = "metaData";

  /**
   * The option property.  It is associated with productions, has a
   * boolean value, and indicates whether a production represents a
   * desugared option.
   */
  public static final String OPTION = "option";

  /**
   * The recursive property.  It is associated with grammars, has a
   * boolean value, and indicates that a grammar contains directly
   * left-recursive generic productions.  It is also associated with
   * productions and indicates that a production is left-recursive.
   */
  public static final String RECURSIVE = "recursive";

  /**
   * The redacted property.  It is associated with productions, has a
   * boolean value, and indicates that a production's body has been
   * removed even though the production really has a body.
   */
  public static final String REDACTED = "redacted";

  /**
   * The repeated property.  It is associated with productions, has a
   * nonterminal value, and indicates that a production represents a
   * desguared repetition whose element value is the value of the
   * nonterminal.
   */
  public static final String REPEATED = "repeated";

  /**
   * The restrict property.  It is associated with productions, has a
   * boolean value, and indicates whether a production restricts the
   * input.
   */
  public static final String RESTRICT = "restrict";

  /**
   * The root property.  It is associated with grammars, has a
   * nonterminal value, and indicates a grammar's real root (i.e., the
   * single public production that directly or indirectly references
   * all other public productions).
   */
  public static final String ROOT = "root";

  /**
   * The split property.  It is associated with productions, has a
   * boolean value, and indicates that the production's alternatives
   * need to be split.  It is also associated with sequences, has a
   * {@link Annotator.IndexPair} value, and indicates how to split the
   * sequence.
   */
  public static final String SPLIT = "split";

  /**
   * The text-only property.  It is associated with productions, has a
   * boolean value, and indicates whether a production is text-only.
   */
  public static final String TEXT_ONLY = "textOnly";

  /**
   * The token property.  It is associated with productions, has a
   * boolean value, and indicates whether a production represents a
   * lexical token.
   */
  public static final String TOKEN = "token";

  /**
   * The transformable property.  It is associated with productions,
   * has a boolean value, and indicates whether a production is a
   * directly left-recursive production that can be transformed into
   * an equivalent right-recursive or -iterative production.
   */
  public static final String TRANSFORMABLE = "transformable";

  /**
   * The voided property.  It is associated with productions, has a
   * boolean value, and indicates whether the production's value has
   * been eliminated.
   */
  public static final String VOIDED = "voided";

  /**
   * Get the value of the duplicates property.
   *
   * @param p The production.
   * @return The list of folded productions.
   */
  @SuppressWarnings("unchecked")
  public static List<String> getDuplicates(Production p) {
    return (List<String>)p.getProperty(DUPLICATES);
  }

  /**
   * Get the value of the formatting property.
   *
   * @param s The sequence.
   * @return The list of bindings.
   */
  @SuppressWarnings("unchecked")
  public static List<Binding> getFormatting(Sequence s) {
    return (List<Binding>)s.getProperty(FORMATTING);
  }

}
