#include <stdio.h>
#include <string.h>

struct partition{
	unsigned char boot_ind;
	unsigned char head;
	unsigned char sector;
	unsigned char cyl;
	unsigned char sys_ind;
	unsigned char end_head;
	unsigned char end_sector;
	unsigned char end_cyl;
	unsigned int start_sec;
	unsigned int num_sec;
};

struct hd_info_struct{
	int flag;
	unsigned int start_sec;
	unsigned int num_sec;
}hd_info[4];

#define HDIMG	"./hdimg"

#define SECSIZE		512
#define BLOCKSIZE	(2 * SECSIZE)

#define BOOTBLOCK	0	/* unit: BLOCKSIZE bytes */
#define BBSIZE		1
#define RESERVED	(BOOTBLOCK + BBSIZE)
#define RVSIZE		2	
#define SUPERBLOCK	(RESERVED + RVSIZE)
#define SBSIZE		1
#define ROOTBLOCK	(SUPERBLOCK + SBSIZE)
#define RBSIZE		1
#define BITBLOCK	(ROOTBLOCK + RBSIZE)
#define BITSIZE		1
#define INODEBLOCK	(BITBLOCK + BITSIZE)
#define IBSIZE		2	
#define DATABLOCK	(INODEBLOCK + IBSIZE)

struct superblock{
	char version[32];
	int di_used;
	int low_fb, high_fb;
}sb;

#define DIRITEM		32
#define FNAME_BYTES	29

#define DIRITEMNUM	(BLOCKSIZE / DIRITEM)

#define FREE	0
#define USED	1

struct dir_item{
	unsigned char flag;	/* FREE OR USED */
	char fname[FNAME_BYTES];
	unsigned short inum;
};

struct dir_item root_block[DIRITEMNUM];

/* from 0 block --> ... */
/* the bit before DATABLOCK is reserved */
unsigned char bitmap[BLOCKSIZE];

struct inode{
	unsigned long attr;	/* Now used for used or free flag */
	unsigned long dev;
	unsigned long ctime;
	unsigned long atime;
	unsigned long mtime;
	unsigned long fsize;
	unsigned short block[20];
};

#define INODE_NUM	(BLOCKSIZE * IBSIZE) / (sizeof(struct inode))
struct inode inodes[INODE_NUM];

FILE * hd_fp;

void init_hd(void);
void close_hd(void);
void get_hd_info(void);
void read_block(int, char *);
void write_block(int, char *);
void init_fs(void);
void flush_fs(void);
void exec_cmd(void);
void do_format(void);
void do_showinfo(void);
void do_copy(char *, char *);
void do_list(void);
void do_cat(char *);

int main(void)
{
	init_hd();	
	get_hd_info();
	init_fs();

	exec_cmd();

	close_hd();
}

void init_hd(void)
{
	hd_fp = fopen(HDIMG, "r+");

	if(hd_fp == NULL){
		fprintf(stderr, "Open hdimg error!");
		exit(0);
	}
}

void close_hd(void)
{
	fclose(hd_fp);
}

void get_hd_info(void)
{
	int i;
	unsigned char buffer[SECSIZE];
	struct partition * par_ptr;

	fread(buffer, SECSIZE, 1, hd_fp);
	par_ptr = (struct partiton *)(&buffer[0x1be]);
	for(i = 0; i < 4; i++){
		if(par_ptr->sys_ind == 0){
			hd_info[i].flag = 0;
			continue;
		}

		hd_info[i].flag = 1;
		hd_info[i].start_sec = par_ptr->start_sec;
		hd_info[i].num_sec = par_ptr->num_sec;
		printf("%d: %x %x %dM!\n", i, par_ptr->sys_ind,
			par_ptr->start_sec, par_ptr->num_sec / 2048);

		par_ptr++;
	}
}

void read_block(int blocknr, char * buffer)
{
	fseek(hd_fp, ((2 * blocknr + hd_info[0].start_sec) * SECSIZE), SEEK_SET);
	fread(buffer, BLOCKSIZE, 1, hd_fp);
}

void write_block(int blocknr, char * buffer)
{
	fseek(hd_fp, ((2 * blocknr + hd_info[0].start_sec) * SECSIZE), SEEK_SET);
	fwrite(buffer, BLOCKSIZE, 1, hd_fp);
}

void init_fs(void)
{
	char buffer[BLOCKSIZE];
	struct superblock * sbptr;

	read_block(SUPERBLOCK, buffer);
	sbptr = (struct supberblock *)buffer;
	memcpy(sb.version, sbptr->version, 32);
	sb.di_used = sbptr->di_used;
	sb.low_fb = sbptr->low_fb;
	sb.high_fb = sbptr->high_fb;
	
	read_block(ROOTBLOCK, (char *)&root_block[0]);

	read_block(BITBLOCK, (char *)bitmap);

	/* ... */
	read_block(INODEBLOCK, (char *)&inodes[0]);
	read_block(INODEBLOCK + 1, (char *)&inodes[16]);
}

void flush_fs(void)
{
	char buffer[BLOCKSIZE];
	struct superblock * sbptr;

	sbptr = (struct supberblock *)buffer;
	memcpy(sbptr->version, sb.version, 32);
	sbptr->di_used = sb.di_used;
	sbptr->low_fb = sb.low_fb;
	sbptr->high_fb = sb.high_fb;
	write_block(SUPERBLOCK, buffer);
	
	write_block(ROOTBLOCK, (char *)&root_block[0]);

	write_block(BITBLOCK, (char *)bitmap);

	/* ... */
	write_block(INODEBLOCK, (char *)&inodes[0]);
	write_block(INODEBLOCK + 1, (char *)&inodes[16]);
}

void exec_cmd(void)
{
	char cmd[1024];
	char * arg0, * arg1, * arg2;

	do{
		printf("\n>");
		fgets(cmd, 1024, stdin);	
		arg0 = strtok(cmd, " \n");
		if(arg0 != NULL){
			if(strcmp(arg0, "format") == 0){
				do_format();
				flush_fs();
			}else if(strcmp(arg0, "showinfo") == 0){
				do_showinfo();
			}else if(strcmp(arg0, "cp") == 0){
				arg1 = strtok(NULL, " \n");
				arg2 = strtok(NULL, " \n");
				if(arg1 == NULL || arg2 == NULL){
					fprintf(stderr, "cp srcfile dstfile\n");
					continue;
				}
				do_copy(arg1, arg2);
			}else if(strcmp(arg0, "quit") == 0){
				break;
			}else if(strcmp(arg0, "ls") == 0){
				do_list();	
			}else if(strcmp(arg0, "cat") == 0){
				arg1 = strtok(NULL, " \n");
				if(arg1 == NULL){
					fprintf(stderr, "cat filename\n");
					continue;
				}
				do_cat(arg1);
			}else if(strcmp(arg0, "help") == 0){
				printf("format\nshowinfo\ncp src dst\nls\ncat\nhelp\nquit\n");
			}else{
				fprintf(stderr, "You have input a unrecognized command!\n");
			}
		}else
			fprintf(stderr, "Please input a valid command!\n");
		
	}while(1);
}

void do_format(void)
{
	strcpy(sb.version, "CNIX FILESYSTEM V1.0");
	sb.di_used = 0;
	sb.low_fb = 0;
	sb.high_fb = (BLOCKSIZE * SBSIZE) * 8 - 1;
	
	memset((void *)&root_block[0], 0, BLOCKSIZE * RBSIZE);
	
	memset((void *)bitmap, 0, BLOCKSIZE * BITSIZE);

	memset((void *)&inodes[0], 0, BLOCKSIZE * IBSIZE);
}

void do_showinfo(void)
{
	printf("Version: %s\n", sb.version);
}

struct dir_item * get_diritem(void)
{
	int i;
	struct dir_item * diptr;

	diptr = NULL;
	for(i = 0; i < DIRITEMNUM; i++){
		diptr = &root_block[i];
		if(diptr->flag == FREE){
			diptr->flag = USED;
			break;
		}
	}

	return diptr;
}

int get_inode(void)
{
	int i;
	struct inode * iptr;

	iptr = NULL;
	for(i = 0; i < INODE_NUM; i++){
		iptr = &inodes[i];
		if(iptr->attr == FREE){
			iptr->attr = USED;
			break;
		}
	}

	if(i == INODE_NUM)
		return -1;

	return i;
}

unsigned short get_block(void)
{
	unsigned short i;
	int index, offset;

	for(i = DATABLOCK; i < BLOCKSIZE * 8; i++){
		index = i / 8;
		offset = i % 8;
		if(!((bitmap[index] >> offset) & USED)){
			bitmap[index] |= USED << offset;
			break;
		}
	}

	if(i == BLOCKSIZE * 8)
		return 0;

	return i;
}

void do_copy(char * srcfile, char * dstfile)
{
	FILE * srcfp;
	char buff[1024];
	int filesize, readbytes;
	struct dir_item * diptr;
	int inum;
	struct inode * iptr;
	unsigned short blocknr;
	int index;

	srcfp = fopen(srcfile, "r");
	if(srcfp == NULL){
		fprintf(stderr, "Srcfile doesn't exist!\n");
		return;
	}

	diptr = get_diritem();
	if(diptr == NULL)
		return;
	strcpy(diptr->fname, dstfile);

	inum = get_inode();
	if(inum == -1)
		return;

	diptr->inum = inum;
	iptr = &inodes[inum];

	index = 0;
	filesize = 0;
	while((readbytes = fread(buff, 1, 1024, srcfp))){
		blocknr = get_block();
		iptr->block[index++] = blocknr;
		write_block(blocknr, buff);
		filesize += readbytes;
	}

	iptr->fsize = filesize;

	fclose(srcfp);

	flush_fs();
}

void do_list(void)
{
	int i;

	for(i = 0; i < DIRITEMNUM; i++){
		if(root_block[i].flag == USED){
			printf("filename: %s	inode number: %d\n",
				root_block[i].fname, root_block[i].inum);
		}
	}
}

void do_cat(char * filename)
{
	FILE * srcfp;
	char buff[BLOCKSIZE];
	int allbytes, readbytes, i;
	struct dir_item * diptr;
	int inum;
	struct inode * iptr;
	unsigned short blocknr;
	int index;

	for(i = 0; i < DIRITEMNUM; i++){
		if(root_block[i].flag == USED 
			&& strcmp(root_block[i].fname, filename) == 0)
			break;
	}

	if(i == DIRITEMNUM)
		return;

	diptr = &root_block[i];
	inum = diptr->inum;

	iptr = &inodes[inum];

	allbytes = iptr->fsize;

	if(allbytes > 1024){
		readbytes = 1024;
		allbytes -= 1024;
	}else{
		readbytes = allbytes;
		allbytes = 0;
	}

	index = 0;
	while(readbytes){
		blocknr = iptr->block[index++];
		read_block(blocknr, buff);
		for(i = 0; i < readbytes; i++)
			printf("%c", buff[i]);

		if(allbytes > 1024){
			readbytes = 1024;
			allbytes -= 1024;
		}else{
			readbytes = allbytes;
			allbytes = 0;
		}
	}
}
