#include <string.h>
#include <asm/io.h>
#include <asm/system.h>
#include <cnix/wait.h>
#include <cnix/kernel.h>
#include <cnix/driver.h>

#define VGA_CRT_INX	0x3d4
#define VGA_CRT_DATA 	0x3d5

#define REG_ORG		12
#define REG_CUR		14	

#define VID_MEM		0xb8000
#define VID_SIZE	0x4000

#define lines		25
#define cols		80

extern struct tty_struct * cur_tty;

extern int put_irq_handler(int irq, fn_t fn);

extern struct tty_struct ttys[];

struct vga_struct vgas[VGA_NUM];
struct vga_struct * cur_vga;

static inline void gotoxy(unsigned int x, unsigned int y);

static void set_vga(int reg, int val)
{
	unsigned int flags;

	save_flags(flags);
	cli();

	outb(reg, VGA_CRT_INX);
	outb(0xff & (val >> 8), VGA_CRT_DATA);
	outb(reg + 1, VGA_CRT_INX);
	outb(0xff & val, VGA_CRT_DATA);

	restore_flags(flags);
}

void select_vga(int index)
{
	if(index < 0 || index > VGA_NUM)
		return;

	cur_vga = &vgas[index];
	cur_tty = cur_vga->tp;
	set_vga(REG_ORG, cur_vga->org);
	set_vga(REG_CUR, cur_vga->cur_pos);
}

static void scroll_up(void)
{
	int i;
	void * src, * dst;
	unsigned char * vidptr;
	struct vga_struct * vp;

	vp = ttys[current->tty].vp;

	if(vp->org + cols * (lines + 1) < vp->limit){
		vp->org += cols;
		set_vga(REG_ORG, vp->org);	
	}else{
		dst = (void *)(VID_MEM + vp->start * 2);
		if(vp->org == vp->start)
			src = (void *)(VID_MEM + vp->org * 2 + cols * 2);
		else
			src = (void *)(VID_MEM + vp->org * 2);
		memcpy(dst, src, (lines - 1) * cols * 2);

		cur_vga->org = vp->start;
		set_vga(REG_ORG, vp->org);
	}
	
	vidptr = (unsigned char *)(VID_MEM + 2 * vp->org + (lines - 1) * cols * 2);
	for (i = 0; i < cols * 2; i += 2)
		vidptr[i] = ' ';
}

static inline void gotoxy(unsigned int x, unsigned int y)
{
	struct vga_struct * vp;

	vp = ttys[current->tty].vp;

	vp->cur_pos = vp->org + cols * y + x;

	set_vga(REG_CUR, vp->cur_pos);
}

void putchar(char  c)
{
	unsigned char * vidmem;
	int x, y, i;
	struct vga_struct * vp;

	vp = ttys[current->tty].vp;
	x = vp->cur_col;
	y = vp->cur_row;

	vidmem = (char *)(VID_MEM + vp->org * 2);

	switch(c){
	case 000:
		return;
	case 007:	/* beep */
		break;
	case '\b':
		if(x > 0)
			x--;
		else
			if(y > 0){
				x = cols - 1;
				y--;
			}

		vidmem[(x + y * cols) * 2] = ' ';

		break;
	case '\n':
		x = 0;
		if(++y >= lines) { /* has gone to bottom of screen */
			scroll_up();
			vidmem = (char *)(VID_MEM + vp->org * 2);

			y--;
		}

		break;
	case 013:	/* CTRL-K */
		break;
	case 014:	/* CTRL-L */
		break;
	case '\r':
		x = 0;
		break;
	case '\t':
#define TAB_SPACES	8
		for(i = 0; i < TAB_SPACES; i++){
			vidmem[(x + y * cols) * 2] = ' ';
			if(x < cols - 1)
				x++;
			else{
				x = 0;

				if(y < lines - 1)
					y++;
				else{
					scroll_up();
					vidmem = (char *)(VID_MEM + vp->org * 2);
				}
			}
		}
	
		break;
	case 033:	/* ESC */
		break;
	default:
		vidmem [(x + cols * y) * 2] = c; 
		if(++x >= cols){
			x = 0;
			if(++y >= lines){
				scroll_up();
				vidmem = (char *)(VID_MEM + vp->org * 2);

				y--;
			}
		}

		break;
	}

	vp->cur_col = x;
	vp->cur_row = y;

	gotoxy(x, y);
}

void puts(char *s)
{
	unsigned char c;

	while ( ( c = *s++ ) != '\0' ) {
		putchar(c);
/*		if ( c == '\n' ) {
			x = 0;
			if ( ++y >= lines ) {
				scroll_up();
				vidmem = (char *)(VID_MEM + cur_vga->org * 2);

				y--;
			}
		} else {
			vidmem [ ( x + cols * y ) * 2 ] = c; 
			if ( ++x >= cols ) {
				x = 0;
				if ( ++y >= lines ) {
					scroll_up();
					vidmem = (char *)(VID_MEM + cur_vga->org * 2);
					y--;
				}
			}
		}
*/
	}
}

unsigned char readchar(void)
{	
	unsigned char ch;
	struct tty_struct * tp;
	struct tty_queue * tq;

	tp = &ttys[current->tty];
	tq = &tp->tq;
	if(tq->tail == tq->head)
		sleep_on(&tp->wait);		

	ch = tq->buf[tq->tail];

	tq->tail = (tq->tail + 1) & (SIZE - 1);
	
	return ch;
}

void do_tty_interrupt(void)
{
	/* check for cur_tty is NULL or not */
	if(cur_tty->tq.tail != cur_tty->tq.head)
		if(cur_tty->wait != NULL)
			wakeup(&cur_tty->wait);
}

void vga_init(int tty)
{
	struct tty_struct * tp;
	struct vga_struct * vp;
	int scr_size;

	if(tty > VGA_NUM - 1)
		return;

	scr_size = (VID_SIZE >> 1) / VGA_NUM;

	tp = &ttys[tty];
	vp = &vgas[tty];

	tp->vp = vp;
	vp->tp = tp;
	
	vp->cur_row = 0;
	vp->cur_col = 0;

	vp->start = tty * scr_size;
	vp->limit = vp->start + scr_size;
	vp->org = vp->start;
	vp->cur_pos = vp->org + vp->cur_row * cols + vp->cur_col;
}
